//--------------------------------------------------------------
// Reverb unit
//  2017/01/31, Copyright (c) 2017 MIKAMI, Naoki
//--------------------------------------------------------------

#ifndef REVERB_UNIT_HPP
#define REVERB_UNIT_HPP

#include "mbed.h"
#include "Array.hpp"
namespace Mikami
{
    // Base class for reverb unit
    class Reverb
    {
    public:
        // Constructor
        Reverb(int delay) : DELAY_(delay), un_(delay)
        {   
            ptr_ = 0;
            Clear();
        }
        
        ~Reverb() {}

        // Execute of filter (pure virtual function)
        virtual float Execute(float x) = 0;

        // Clear internal delay elements
        void Clear()
        {   for (int n=0; n<DELAY_; n++) un_[n] = 0; }

    protected:
        float Get() { return un_[ptr_]; }

        void Set(float x)
        {
            un_[ptr_] = x;   
            if (++ptr_ >=  DELAY_) ptr_ = 0;
        }

    private:
        const int DELAY_;
        Array<float> un_;   // for delay
        int ptr_;

        // disallow copy constructor and assignment operator
        Reverb(const Reverb&);
        Reverb& operator=(const Reverb&);
    };

    // Reverb unit using comb filter
    class CombFilter : public Reverb
    {
    public:
        // Constructor
        CombFilter(float g, int delay)
            : Reverb(delay), G0_(g) {}

        // Execute comb filter 
        virtual float Execute(float x)
        {
            float yn = Reverb::Get();
            Reverb::Set(x + G0_*yn);
            return yn;
         }

    private:
        const float G0_;
    };

    // Reverb unit using allpass filter
    class AllPassFilter : public Reverb
    {
    public:
        // Constructor
        AllPassFilter(float g, int delay)
            : Reverb(delay), G0_(g) {}

        // Execute allpass filter 
        virtual float Execute(float x)
        {
            float un = x + G0_*Reverb::Get();
            float yn = -G0_*un + Reverb::Get();
            Reverb::Set(un);
            return yn;
        }

    private:
        const float G0_;
    };
}
#endif  // REVERB_UNIT_HPP
