//--------------------------------------------------------------
//  Weaver 変調器
//      入力信号の帯域：             0.1 kHz ---- 9.9 kHz
//      帯域中央の周波数：           5.0 kHz
//      低域通過フィルタの遮断周波数： 4.9 kHz

// 2017/01/31, Copyright (c) 2017 MIKAMI, Naoki
//--------------------------------------------------------------

#ifndef WEAVER_MODULATOR_HPP
#define WEAVER_MODULATOR_HPP

#include "IIR_Cascade.hpp"
#include "Coefficients.hpp"  // Coeffisients of LPF and DC-cut filter
#include "Array.hpp"
using namespace Mikami;

class FrqShifter : public SignalProcessing
{
public:
    FrqShifter(float fS)
        : lpfC_(WEAVER_ORDER_, weaver_g0Lpf_, weaver_ck_),
          lpfS_(WEAVER_ORDER_, weaver_g0Lpf_, weaver_ck_),
          dcCut_(weaver_c1_),
          FS_(fS), F_B_(5000)
    {
        lpfC_.Clear();
        lpfS_.Clear();

        dPhi1_ = F_B_*PI2_/FS_;
        dPhi2_ = dPhi1_;
        phi1_ = 0;
        phi2_ = 0;
    }

    virtual float Execute(float xn1, float xn2)
    {
        float xn = (xn1 + xn2)*0.5f;
        xn = dcCut_.Execute(weaver_g0_*xn); // DC 成分除去

        float mpyC = xn*cosf(phi1_);
        float mpyS = xn*sinf(phi1_);

        // LPF
        float mpyC_Lpf = lpfC_.Execute(mpyC);
        float mpyS_Lpf = lpfS_.Execute(mpyS);

        float mpyC_C = mpyC_Lpf*cosf(phi2_);
        float mpyS_S = mpyS_Lpf*sinf(phi2_);

        float yn = 2.0f*(mpyC_C + mpyS_S);

        phi1_ = phi1_ + dPhi1_;
        if (phi1_ > PI2_) phi1_ = phi1_ - PI2_;
        phi2_ = phi2_ + dPhi2_;
        if (phi2_ > PI2_) phi2_ = phi2_ - PI2_;

        return yn;
    }

    // 周波数のシフト量を設定
    void SetFrequensy(float fShift)
    {   dPhi2_ = (fShift + F_B_)*PI2_/FS_; }

private:
    // Weaver 変調器で使う低域通過フィルタ
    IirCascade lpfC_;
    IirCascade lpfS_;
    // 直流分除去フィルタ
    Biquad dcCut_;

    static const float PI2_ = 3.1415926536f*2;
    const float FS_;
    const float F_B_;      // 中心周波数

    float phi1_, dPhi1_;
    float phi2_, dPhi2_;
};
#endif  // WEAVER_MODULATOR_HPP
