//----------------------------------------------------------------------
//  TIM7 の Update event (UEV) 割込みを使って，Ticker と同じような機能の
//  ほかに，割込みの時間間隔を設定するメンバ関数を追加した行うクラス
//
//  動作を確認したボード： NUCLEO-F446RE, DISCO-F746
//
//    mbed 公式ライブラリの Ticker クラスは，割込みの時間間隔を変更する場合は，
//  メンバ関数 attach() を呼び出す必要がある．しかし，attach() を呼び出すと，
//  割込みの時間間隔設定以外の処理も行われるため，アプリケーションによっては
//  割込み時間間隔が変更される際に，動作が不自然になる場合がある．
//    この MyTicker7 クラスは，TIM7 の ARR 設定の際に，ARR バッファを使う
//  ように設定し，UEV 発生までは ARR の値を書き換えないようにした．
//
//  2018/06/24, Copyright (c) 2018 MIKAMI, Naoki
//----------------------------------------------------------------------

#ifndef MYTICKER14_F446_746_HPP
#define MYTICKER14_F446_746_HPP

#include "mbed.h"

#ifndef TIM7
#error "This MCU has No TIM7."
#endif

namespace Mikami
{
    class MyTicker7
    {
    public:
        MyTicker7();
        // 割込みサービスルーチンの割り当てと割り込み周期（μs 単位）の設定
        void Attach_us(void (*Func)(), uint16_t time);
        // 割込みサービスルーチンの割り当てと割り込み周期（ms 単位）の設定
        void Attach_ms(void (*Func)(), uint16_t time);
        // 割込み停止
        void Disable() { tim_->DIER &= ~TIM_DIER_UIE; }
        // 割込み開始
        void Enable() { tim_->DIER |= TIM_DIER_UIE; }
        // 割り込み周期（μs 単位）の設定
        void SetPeriod_us(uint16_t time);
        // 割り込み周期（ms 単位）の設定
        void SetPeriod_ms(uint16_t time);

    private:
        // 割り込みサービス･ルーチンの中で実行される関数のポインタ
        static void (*fp)();
        // タイマに対応する構造体のポインタ
        static TIM_TypeDef* const tim_;
        // 割り込みサービス･ルーチン
        static void Isr();
        // タイマ設定の共通の処理
        void SetCommon(void (*Func)(), uint16_t time);
        
        uint32_t maxClock_;     // TIM7 を駆動するクロック周波数
        
        // 割り込みサービス･ルーチンの準備と割り込み許可
        void SetIsr(void (*Func)(), uint16_t time);
    };
}
#endif  // MYTICKER14_F446_746_HPP

