//----------------------------------------------------------------------
//  TIM7 の Update event (UEV) 割込みを使って，Ticker と同じような機能の
//  ほかに，割込みの時間間隔を設定するメンバ関数を追加した行うクラス
//
//  2018/06/24, Copyright (c) 2018 MIKAMI, Naoki
//----------------------------------------------------------------------

#include "MyTicker7.hpp"

namespace Mikami
{
    MyTicker7::MyTicker7()
    {
        // TIM7 にクロック供給．"stm32f4xx_hal_rcc_ex.h" 参照
        __HAL_RCC_TIM7_CLK_ENABLE();
        // クロックの更新（念のため）．"system_stm32f4xx.h" 参照
        SystemCoreClockUpdate();
#if defined(STM32F7)        
        if ((RCC->DCKCFGR1 & RCC_DCKCFGR1_TIMPRE) == RCC_DCKCFGR1_TIMPRE)
#elif defined(STM32F4)
        if ((RCC->DCKCFGR & RCC_DCKCFGR_TIMPRE) == RCC_DCKCFGR_TIMPRE)
#else
#error "This MCU is neither STM32F7 nor STM32F4."
#endif
            maxClock_ = SystemCoreClock;
        else
            maxClock_ = SystemCoreClock/2;
    }

    // 割込みサービスルーチンの割り当てと割り込み周期（μs 単位）の設定
    void MyTicker7::Attach_us(void (*Func)(), uint16_t time)
    {
        SetPeriod_us(time);
        SetCommon(Func, time);
    }

    // 割込みサービスルーチンの割り当てと割り込み周期（ms 単位）の設定
    void MyTicker7::Attach_ms(void (*Func)(), uint16_t time)
    {
        SetPeriod_ms(time);
        SetCommon(Func, time);
    }

    // 割り込み周期（μs 単位）の設定
    void MyTicker7::SetPeriod_us(uint16_t time)
    {
        tim_->ARR = time - 1;               // Auto-reload
        tim_->PSC = maxClock_/1000000 - 1;  // Prescaler
    }

   // 割り込み周期（ms 単位）の設定
    void MyTicker7::SetPeriod_ms(uint16_t time)
    {
        tim_->ARR = 10*time - 1;            // Auto-reload
        tim_->PSC = maxClock_/10000 - 1;    // Prescaler
    }
    
    // 割り込みサービス･ルーチン
    void MyTicker7::Isr()
    {
        tim_->SR &= ~TIM_SR_UIF;   // Update 割込みをクリア
        fp();       // Attach_us(), Attach_ms() の引数として渡された関数を実行
    }
    
    // 割り込みサービス･ルーチンの準備と割り込み許可
    void MyTicker7::SetIsr(void (*Func)(), uint16_t time)
    {
        fp = Func;
        NVIC_SetVector(TIM7_IRQn, (uint32_t)Isr);
        NVIC_EnableIRQ(TIM7_IRQn);
    }

    // タイマ設定の共通の処理
    void MyTicker7::SetCommon(void (*Func)(), uint16_t time)
    {
        // CR1 の ARPE ビットを 1 にすることで，ARR の更新を
        // Update event (UVE) に合わせて行うように設定できる
        // リファレンス･マニュアル (RM0390) p.587 参照
        tim_->CR1 = TIM_CR1_CEN             // Counter 有効
                  | TIM_CR1_ARPE;           // ARR のバッファ有効
        tim_->DIER |= TIM_DIER_UIE;         // Update 割込み有効

        SetIsr(Func, time);
    }

    void (*MyTicker7::fp)();
    TIM_TypeDef* const MyTicker7::tim_ = TIM7;
}
