//---------------------------------------------------------------------
//  オシロスコープ (Nucleo-F446RE 用)
//
//      ● ST-Link Firmware の V2.J31.M21 で動作確認
//      ● ST-Link Firmware のアップグレードには stsw-link07.zip
//        に含まれている "ST-LinkUpgrade.exe" を使う 
//
//      ● PC 側のプログラム： "F446_Oscilloscope"
//      ● ボーレート： 460800 baud
//      ● 受信データの文字列の終了マーク： "\r"
//
//      ● 入力  A0: チャンネル1，A1 チャンネル2
//
//  2020/12/24, Copyright (c) 2020 MIKAMI, Naoki
//---------------------------------------------------------------------

#include "mbed.h"
#include <string>
#include "Array.hpp"
#include "DSP_AdcDualIntr.hpp"
#include "InputBuffer.hpp"
#include "Xfer.hpp"
using namespace Mikami;

#ifndef __STM32F446xx_H
#error "Use Nucleo-F446RE"
#endif

const int N_FRAME_ = 1000;   // １フレーム当たり標本化するデータ数

DspAdcDualIntr myAdc_(1000, A0, A1);    // 起動時の標本化周波数： 1 MHz
InputBuffer<float> buf_(N_FRAME_);      // AD の結果を保存するバッファ

// ADC 変換終了割り込みに対する割り込みサービス･ルーチン
void AdcIsr()
{
    float sn1, sn2;
    myAdc_.Read(sn1, sn2);
    buf_.Store(sn1, sn2);       // バッファへ格納
    if (buf_.IsFullNext())      // バッファが満杯になったら ADC 割り込みを禁止する
        myAdc_.DisableAdcIntr();
}

int main()
{
    SerialRxTxIntr rxTx(32, 115200*4);  // PC との通信用
    Xfer tx(rxTx, 2*N_FRAME_);          // PC に転送するためのオブジェクトの生成
    Array<float> sn(2*N_FRAME_);        // 転送するデータ

    NVIC_SetPriority(ADC_IRQn, 0);      // AD変換終了割り込みの優先度が最高
    NVIC_SetPriority(USART2_IRQn, 1);

    bool ready = false;     // データの変換終了で true
    bool okGo = false;      // "GO" を受信したら true

    myAdc_.SetIntrVec(&AdcIsr); // AD変換終了割り込みの割り当て
    while (true)
    {
        // PC からのコマンドの解析
        if (rxTx.IsEol())       // 受信バッファのデータが有効になった場合の処理
        {
            string str = rxTx.GetBuffer();
            if (str == "Oscilloscope")
                rxTx.TxString("ACK\n");     // PC からの "Oscilloscope" 受信に対して
                                            // "ACK" を送信する
            else if (str.substr(0, 2) == "GO")
            {
                // str の内容
                // [0]      'G'
                // [1]      'O'
                // [2..5]    標本化周波数

                int fs = atoi(str.substr(2, 4).c_str());    // 標本化周波数
                myAdc_.SetFs(fs);       // 標本化周波数再設定

                okGo = true;            // データの転送要求あり
            }
        }

        if (buf_.IsFull())  // 入力データが満杯の場合，以下の処理を行う
        {
            for (int n=0; n<N_FRAME_; n++)
            {
                sn[n] = buf_.Get(0, n);
                sn[n+N_FRAME_] = buf_.Get(1, n);
            }
            tx.Convert(sn);
            ready = true;           // データの変換終了
        }

        // 転送要求があり，データの変換が終了している場合に，データを PC へ転送する
        if (okGo && ready)
        {
            tx.ToPC();      // データを PC へ転送
            ready = false;
            okGo = false;

            myAdc_.EnableAdcIntr(); // ADC 割り込みを有効にする
        }
    }
}