//---------------------------------------------------------------------
//  FFT によるスペクトルアナライザ，白色雑音発生器付き (Nucleo-F446RE 用)
//
//      ● ST-Link Firmware の V2.J37.M26 で動作確認
//
//      ● ST-Link Firmware のアップグレードには stsw-link007_V2-37-26.zip
//        に含まれている "ST-LinkUpgrade.exe" を使う 
//
//      ● PC 側のプログラム： "F446_FFT_Analyzer"
//      ● ボーレート： 460800 baud
//      ● 受信データの文字列の終了マーク： "\r"
//
//      ● 入力：  A1
//      ● 白色雑音の出力：A2
//
//  2021/01/08, Copyright (c) 2021 MIKAMI, Naoki
//---------------------------------------------------------------------

#include "mbed.h"
#include <string>
#include "Array.hpp"
#include "DSP_AdcIntr.hpp"
#include "DSP_Dac.hpp"
#include "FFT_Analyzer.hpp"
#include "DoubleBuffer.hpp"
#include "Coefs_IIR_LP.hpp" // 縦続形 IIR フィルタの係数
#include "IirCascade.hpp"   // 縦続形 IIR フィルタ
#include "MSeq16.hpp"       // M 系列発生器
#include "Xfer.hpp"
using namespace Mikami;

const int N_FFT_ = 1024;            // FFT の点数
const int N_FRAME_ = N_FFT_;        // １フレーム当たり標本化するデータ数
const int N_SPC_ = N_FFT_/2 + 1;    // 有効なスペクトルの点数
const int RATIO_ = 10;              // オーバーサンプリングの倍率

DoubleBuffer<float, N_FRAME_> buf_(0);  // AD の結果を保存するバッファ
DspAdcIntr myAdc_(10*RATIO_, A1);       // 標本化周波数： 100 kHz
DspDac myDac;
IirCascade df1_(ORDER1_, CK1_, G01_);   // ダウンサンプリング用 Anti-alias フィルタ
IirCascade df2_(ORDER2_, CK2_, G02_);   // 白色雑音発生用
MSeq16 mSeq_;

// ADC 変換終了割り込みに対する割り込みサービス･ルーチン
void AdcIsr()
{
    static int count = 0;

    float xn = myAdc_.Read();

    float noise = df2_.Execute(mSeq_.Execute());
    myDac.Write(0.8f*noise);    // 白色雑音出力

    float yn = df1_.Execute(xn);    // ダウンサンプリング用 Anti-alias フィルタの実行

    if (++count >= RATIO_)
    {
        buf_.Store(yn);         // ダウンサンプリングされたデータをバッファへ格納
        count = 0;
        buf_.IfFullSwitch();    // バッファが満杯であればバッファを切り替える
    }
}

int main()
{
    SerialRxTxIntr rxTx(32, 460800);    // PC との通信用
    Xfer tx(rxTx, N_SPC_);              // PC に転送するためのオブジェクトの生成
    FftAnalyzer analyzer(N_FFT_);       // FFT によるスペクトル解析オブジェクトの生成

    Array<float> sn(N_FFT_, 0.0f);      // スペクトル解析の対象となるデータ
    Array<float> absFt(N_SPC_);         // 解析結果：スペクトルの絶対値

    NVIC_SetPriority(ADC_IRQn, 0);      // AD変換終了割り込みの優先度が最高
    NVIC_SetPriority(USART2_IRQn, 1);

    bool ready = false; // スペクトルの計算終了で true
    bool okGo = false;  // "GO" を受信したら true
    float magn = 1;     // 転送する際の倍率

    myAdc_.SetIntrVec(&AdcIsr); // AD変換終了割り込みの割り当て
    while (true)
    {
        // PC からのコマンドの解析
        if (rxTx.IsEol())      // 受信バッファのデータが有効になった場合の処理
        {
            string str = rxTx.GetBuffer();
            if (str == "FFTAnalyzer")
                rxTx.TxString("ACK\n"); // PC からの "FFTAnalyzer" に対して "ACK" を送信
            if (str.substr(0, 2) == "GO")
            {
                okGo = true;  // データの転送要求あり
                float x = (float)atoi(str.erase(0, 2).c_str());
                magn = powf(10, x/20.0f);
            }
        }

        if (buf_.IsFull())  // 入力データが満杯の場合，以下の処理を行う
        {
            for (int n=0; n<N_FRAME_; n++) sn[n] = buf_.Get(n);
            analyzer.Execute(sn, absFt);    // スペクトル解析の実行
            tx.Convert(absFt, magn);        // スペクトル解析の結果を転送する形式に変換
            ready = true;                   // スペクトル解析終了
        }

        // 転送要求がありスペクトル解析が終了している場合にデータを PC へ転送する
        if (okGo && ready)
        {
            tx.ToPC();      // データを PC へ転送
            ready = false;
            okGo = false;
        }
    }
}