//------------------------------------------------------------
//  AD Conversion by polling using ADC2 or ADC3 on STM32F446
//  ---- Header ----
//
//  STM32F446 の ADC2, ADC3 を使ってポーリングによりアナログ信号を
//  入力するクラス（ヘッダ）
//      A0 (PA_0) :  ADC2 CH0
//      A1 (PA_1) :  ADC3 CH1
//
//  2017/02/16, Copyright (c) 2017 MIKAMI, Naoki
//------------------------------------------------------------

#include "mbed.h"

#ifndef STM32F446xx
#error Select NUCLEO-F446RE.
#endif

#include "F446_DAC_Single.hpp"

#ifndef F446_ADC_SINGLE_HPP
#define F446_ADC_SINGLE_HPP

namespace Mikami
{
    class AdcSingle
    {
    public:
        // Constructor
        //      frequency: 標本化周波数
        explicit AdcSingle(PinName pin, int frequency);
        
        virtual ~AdcSingle() {}

        // -1.0f <= ad1, ad2 <= 1.0f
        //      ad1: left, ad2: right
        virtual float Read();

        // 0 <= ad1, ad2 <= 4095
        //      ad1: left, ad2: right
        virtual uint16_t ReadUint();

    protected:
        ADC_TypeDef *adc_;

        float ToFloat(uint16_t x)
        {   return AMP_*(x - 2048); }
    
    private:
        static const float AMP_ = 1.0f/2048.0f;

        void (AdcSingle::*fpWait)();
        
        // AD 変換が完了するまで待つ
        void WaitDone2() { while ((ADC2->SR & ADC_SR_EOC) != ADC_SR_EOC); }
        void WaitDone3() { while ((ADC3->SR & ADC_SR_EOC) != ADC_SR_EOC); }

        // AD 変換器の外部トリガに使うタイマ (TIM8) の設定
        void SetTim8(int frequency);

        // for inhibition of copy constructor
        AdcSingle(const AdcSingle&);
        // for inhibition of substitute operator
        AdcSingle& operator=(const AdcSingle&);     
    };
}
#endif  // F446_ADC_SINGLE_HPP
