//-----------------------------------------------------------
//  出力を 4 倍にアップサンプリングするクラス（ヘッダ）
//  Nucleo-F446RE 専用
//  補間処理で使うフィルタとして，縦続形構成の IIR フィルタを使用
//
//      入力端子： A0 (PA_0) -- デフォルト
//                A1 (PA_1) -- Start() で指定可能
//      出力端子： A2 (PA_4) 
//
//  使用している割り込みと優先順位
//      ADC_IRQn    優先順位： 0, AD 変換終了割り込み
//      EXTI4_IRQn  優先順位： 1, ソフトウェア割り込みとして使用
//
//  2018/10/29, Copyright (c) 2018 MIKAMI, Naoki
//-----------------------------------------------------------

#include "mbed.h"
#include "F446_ADC.hpp"
#include "F446_DAC.hpp"
#include "IIR_Cascade.hpp"

#ifndef F446_MULTIRATE_SWI_HPP
#define F446_MULTIRATE_SWI_HPP

namespace Mikami
{
    class F446_MultirateSWI
    {
    public:
        // コンストラクタ
        //      コンストラクタの引数を与えない場合はデフォルトのフィルタを使用
        //      order = 0 にすると，補間フィルタなしの状態になる
        F446_MultirateSWI(int order = -1, const Biquad hk[] = NULL,
                          float g0 = 1);
        
        virtual ~F446_MultirateSWI()
        {
            delete adc_;
            delete interpolator_;
        }

        // 標本化の実行開始
        //      frequency   入力の標本化周波数
        //      Func        信号処理を行う割り込みサービス･ルーチン
        //      pin         入力ピン（デフォルトは A0．A1 の指定も可）
        void Start(int frequency, void (*Func)(), PinName pin = A0);

        // AD変換の結果を取り出す
        float Input() { return xn_; }
        
        // 補間用フィルタを実行し，処理結果を出力用バッファへ書き込む
        void Output(float yn);          
        
    private:
        static const int FACTOR_ = 4;   // アップサンプリング倍率：4 倍
                                        // この倍率は 2 のべき乗にすること
        static const int MASK_FACTOR_ = FACTOR_ - 1;
        static const int MASK_BUF_ = 2*FACTOR_ - 1;

        static AdcF446 *adc_;       // AD変換器のオブジェクトのポインタ
        static DacF446 dac_;        // DA変換器のオブジェクト
        
        static Array<float> buf_;   // DA変換器に出力するデータ用バッファ
        int indexW_;                // buf_ へ書き込む際のインデックス
        static int indexR_;         // buf_ から読み出す際のインデックス
        static float xn_;           // AD変換器から入力されたデータ

        // 補間用フィルタ用のメンバ
        static const Biquad HK_[];  // biquad 部の係数
        static const float G0_;     // 利得定数
        IirCascade *interpolator_;  // 補間用フィルタのポインタ

        // 引数を 0 ～ (アップサンプリング倍率-1) の間でカウントアップ
        static inline int ModCounter(int &index)
        {
            index = ++index & MASK_BUF_;
            return index;
        }

        // ADC 変換終了割り込みに対する割り込みサービス･ルーチン
        static void AdcIsr();
    };
}
#endif  // F446_MULTIRATE_SWI_HPP