//-----------------------------------------------------------
//  出力を 4 倍にアップサンプリングするクラス（ヘッダ）
//  Nucleo-F446RE 専用
//  補間処理で使うフィルタとして，縦続形構成の IIR フィルタを使用
//
//      入力端子： A0 (PA_0) -- デフォルト
//                A1 (PA_1) -- Start() で指定可能
//      出力端子： A2 (PA_4) 
//
//  ＜注意＞
//      ● 重い信号処理を実行する場合は正常に動かない場合がある．
//      ● メンバ関数 Start() を実行した後に，printf() のような重い
//        処理を実行すると正常に動かないことがある．
//
//  2018/10/11, Copyright (c) 2018 MIKAMI, Naoki
//-----------------------------------------------------------

#include "mbed.h"
#include "F446_ADC.hpp"
#include "F446_DAC.hpp"
#include "IIR_Cascade.hpp"

#ifndef F446_MULTIRATE_HPP
#define F446_MULTIRATE_HPP

namespace Mikami
{
    class F446_Multirate
    {
    public:
        // コンストラクタ
        //      コンストラクタの引数を与えない場合はデフォルトのフィルタを使用
        //      order = 0 にすると，補間フィルタなしの状態になる
        F446_Multirate(int order = -1, const Biquad hk[] = NULL,
                       float g0 = 1);
        
        virtual ~F446_Multirate()
        {
            delete adc_;
            delete interpolator_;
        }

        // 標本化の実行開始
        //      frequency   入力の標本化周波数
        //      pin         入力ピン（デフォルトは A0．A1 の指定も可）
        void Start(int frequency, PinName pin = A0);

        // AD変換の結果を取り出す
        float Input();
        
        // 補間用フィルタを実行し，処理結果を出力用バッファへ書き込む
        void Output(float yn);          
        
    private:
        static const int FACTOR_ = 4;   // アップサンプリング倍率：4 倍
                                        // この倍率は 2 のべき乗にすること
        static const int MASK_FACTOR_ = FACTOR_ - 1;
        static const int MASK_BUF_ = 2*FACTOR_ - 1;

        static AdcF446 *adc_;       // AD変換器のオブジェクトのポインタ
        static DacF446 dac_;        // DA変換器のオブジェクト
        
        static Array<float> buf_;   // DA変換器に出力するデータ用バッファ
        int indexW_;                // buf_ へ書き込む際のインデックス
        static int indexR_;         // buf_ から読み出す際のインデックス
        static float xn_;           // AD変換器から入力されたデータ
        static __IO bool okIn_;     // AD変換されたデータが使える場合に true となる

        // 補間用フィルタ用のメンバ
        static const Biquad HK_[];  // biquad 部の係数
        static const float G0_;     // 利得定数
        IirCascade *interpolator_;  // 補間用フィルタのポインタ

        // 引数を 0 ～ (アップサンプリング倍率-1) の間でカウントアップ
        static inline int ModCounter(int &index)
        {
            index = ++index & MASK_BUF_;
            return index;
        }
        // ADC 変換終了割り込みに対する割り込みサービス･ルーチン
        static void AdcIsr();
    };
}
#endif  // F446_MULTIRATE_HPP
