//----------------------------------------------------------------
//  AD Conversion by interrupt using ADC2 or ADC3 on STM32F446
//  ---- Header ----
//
//  STM32F446 の ADC2 または ADC3 を使って割込みによりアナログ信号を
//  入力するクラス ― マルチ･レート処理用（ヘッダ）
//
//      選択可能な入力端子：
//          A0 (PA_0) :  ADC2 CH0 ---- デフォルト
//          A1 (PA_1) :  ADC3 CH1
//
//  2018/04/16, Copyright (c) 2018 MIKAMI, Naoki
//----------------------------------------------------------------

#include "mbed.h"

#ifndef STM32F446xx
#error Not NUCLEO-F446RE.
#endif

#include "F446_DAC.hpp"

#ifndef F446_ADC_SINGLE_HPP
#define F446_ADC_SINGLE_HPP

namespace Mikami
{
    class AdcF446
    {
    public:
        // コンストラクタ
        //      frequency: 標本化周波数
        AdcF446(int frequency, PinName pin = A0);
        
        virtual ~AdcF446() {}

        // -1.0f <= AD変換された値 < 1.0f
        //      ad1: left, ad2: right
        float Read() { return ToFloat(adc_->DR); }

        // 割込みベクタを設定し，ADC 割込みを有効にする
        void SetIntrVec(void (*Func)());
        
        // ADC 割込みを無効にする
        void DisableAdcIntr()
        {   NVIC_DisableIRQ(ADC_IRQn); }

    private:
        static const float AMP_ = 1.0f/2048.0f;
        ADC_TypeDef *adc_;

        float ToFloat(uint16_t x) { return AMP_*(x - 2048); }

        // AD 変換器の外部トリガに使うタイマ (TIM8) の設定
        void SetTim8(int frequency);

        // for inhibition of copy constructor
        AdcF446(const AdcF446&);
        // for inhibition of substitute operator
        AdcF446& operator=(const AdcF446&);     
    };
}
#endif  // F446_ADC_SINGLE_HPP
