//-----------------------------------------------------------
//  出力を 4 倍にアップサンプリングするクラス（ヘッダ）
//  Nucleo-F446RE 専用
//  補間処理で使うフィルタとして，直線位相 FIR フィルタを使用
//
//      入力端子： A0 (PA_0) -- デフォルト
//                A1 (PA_1) -- Start() で指定可能
//      出力端子： A2 (PA_4) 
//
//  2018/07/03, Copyright (c) 2018 MIKAMI, Naoki
//-----------------------------------------------------------

#include "mbed.h"
#include "F446_ADC.hpp"
#include "F446_DAC.hpp"
#include "Array.hpp"

#ifndef F446_LINEARPHASE_HPP
#define F446_LINEARPHASE_HPP

namespace Mikami
{
    class F446_LinearPhase
    {
    public:
        // コンストラクタ
        //      コンストラクタの引数を与えない場合はデフォルトのフィルタを使用
        //      order = 0 にすると，補間フィルタなしの状態になる
        F446_LinearPhase(int order = 72, const float hk1[] = NULL,
                         const float hk2[] = NULL, const float hk3[] = NULL);
        
        virtual ~F446_LinearPhase() { delete adc_; }

        // 標本化の実行開始
        //      frequency   入力の標本化周波数
        //      pin         入力ピン（デフォルトは A0．A1 の指定も可）
        void Start(int frequency, PinName pin = A0);

        // AD変換の結果を取り出す
        float Input();
        
        // 補間用フィルタを実行し，処理結果を出力用バッファへ書き込む
        void Output(float yn);          
        
    private:
        static const int FACTOR_ = 4;   // アップサンプリング倍率：4 倍
                                        // この倍率は 2 のべき乗にすること
        static const int MASK_FACTOR_ = FACTOR_ - 1;
        static const int MASK_BUF_ = 2*FACTOR_ - 1;

        static AdcF446 *adc_;       // AD変換器のオブジェクトのポインタ
        static DacF446 dac_;        // DA変換器のオブジェクト
        
        static Array<float> buf_;   // DA変換器に出力するデータ用バッファ
        static int indexR_;         // buf_ から読み出す際のインデックス
        static float xn_;           // AD変換器から入力されたデータ
        static __IO bool okIn_;     // AD変換されたデータが使える場合に true となる
        int indexW_;                // buf_ へ書き込む際のインデックス
        
        // 補間用フィルタ用
        const int ORDER_;           // 次数
        const int FIR_COUNT_;       // FIR フィルタのループの数
        const int CENTER_;          // 補間処理をしない信号の位置
        Array<float> un_;           // FIR フィルタの遅延器に対応するバッファ
        Array<float> h1_, h2_, h3_; // FIR フィルタの係数

        // 補間用フィルタ用の係数（デフォルト）
        static const float HK1_[], HK2_[], HK3_[];

        // 引数を 0 ～ (アップサンプリング倍率-1) の間でカウントアップ
        static inline int ModCounter(int &index)
        {
            index = ++index & MASK_BUF_;
            return index;
        }
        
        // ADC 変換終了割り込みに対する割り込みサービス･ルーチン
        static void AdcIsr();
        
        // 補間用 FIR フィルタ
        float Interpolator(float hk[]);
    };
}
#endif  // F446_LINEARPHASE_HPP
