//--------------------------------------------------------
//  Class for buit-in dual DAC on STM32F446 ---- Header
//      TIM3 is used for clock to external SCF
//
//  STM32F446 内蔵の DAC 用のクラス（ヘッダ）
//      TIM3 を外付けの SCF のクロックとして使用
//      A2  (PA_4)： 左
//      D13 (PA_5)： 右
//
//  2017/02/19, Copyright (c) 2017 MIKAMI, Naoki
//--------------------------------------------------------

#include "mbed.h"

#ifndef STM32F446xx
#error Select NUCLEO-F446RE.
#endif

#ifndef F446_DAC_DUAL_HPP
#define F446_DAC_DUAL_HPP

namespace Mikami
{
    class DacDual
    {
    public:
        // Constructor
        DacDual() : da1_(PA_4), da2_(PA_5)
        {
            DAC->CR = DAC_CR_EN1 | DAC_CR_TEN1 | DAC_CR_TSEL1 |
                      DAC_CR_EN2 | DAC_CR_TEN2 | DAC_CR_TSEL2;
        }

        virtual ~DacDual() {}

        // -1.0f <= data1, data2 <= 1.0f
        //      data1: left, data2: right
       void Write(float data1, float data2)
        {   WriteDac(ToUint16(data1), ToUint16(data2)); }

        // 0 <= data1, data2 <= 4095
        //      data1: left, data2: right
        void Write(uint16_t data1, uint16_t data2)
        {   WriteDac(__USAT(data1, BIT_WIDTH_),
                     __USAT(data2, BIT_WIDTH_)); }
        
        // Set TIM3 for clock of switched-capacitor filter
        void ScfClock(uint32_t clock);

    private:
        static const uint32_t SWTRIG_ = DAC_SWTRIGR_SWTRIG1
                                      | DAC_SWTRIGR_SWTRIG2;
        static const int BIT_WIDTH_ = 12;
        AnalogOut da1_, da2_;

        // Write dual-channel data
        void WriteDac(uint16_t val1, uint16_t val2)
        {
            DAC->DHR12RD = __PKHBT(val1, val2, 16);
            DAC->SWTRIGR = SWTRIG_;
        }

        // Saturate float to an unsigned 16-bit type
        uint16_t ToUint16(float val)
        {
            return __USAT((val + 1.0f)*2047.0f, BIT_WIDTH_);
        }

        // for inhibition of copy constructor
        DacDual(const DacDual&);
        // for inhibition of substitute operator
        DacDual& operator=(const DacDual&);     
    };
}
#endif  // F446_DAC_DUAL_HPP
