//----------------------------------------------------------
//  Simultanuous AD Conversion by polling using
//  ADC2 and ADC3 on STM32F446
//
//  STM32F446 の ADC2, ADC3 を使って同時に AD 変換を開始し，
//  ポーリングによりアナログ信号を入力するクラス
//      A0 (PA_0) :  ADC2 CH0
//      A1 (PA_1) :  ADC3 CH1
//
//  2017/02/16, Copyright (c) 2017 MIKAMI, Naoki
//----------------------------------------------------------

#include "F446_ADC.hpp"

namespace Mikami
{
    AdcDual::AdcDual(int frequency)
    {
        // PA0, PA1 を ADC 入力として使うための設定
        __HAL_RCC_GPIOA_CLK_ENABLE();
        GPIO_InitTypeDef gpioInit;
        gpioInit.Pin = GPIO_PIN_0 | GPIO_PIN_1;
        gpioInit.Mode = GPIO_MODE_ANALOG;
        gpioInit.Pull = GPIO_NOPULL;
        gpioInit.Speed = GPIO_SPEED_FREQ_VERY_HIGH;
        HAL_GPIO_Init(GPIOA, &gpioInit);

        // ADC2 の設定 (入力ポート：PA0)
        __HAL_RCC_ADC2_CLK_ENABLE();
        ADC2->CR1 = 0x0;
        ADC2->CR2 = ADC_EXTERNALTRIGCONVEDGE_RISING // 外部トリガの立ち上がりで開始される
                  | ADC_EXTERNALTRIGCONV_T8_TRGO    // 外部トリガ： Timer8 TRGO event
                  | ADC_CR2_ADON;                   // ADC を有効にする
        ADC2->SQR3 = 0x0;                           // CH0 を使う

        // ADC3 の設定 (入力ポート：PA1)
        __HAL_RCC_ADC3_CLK_ENABLE();
        ADC3->CR1 = 0x0;
        ADC3->CR2 = ADC_EXTERNALTRIGCONVEDGE_RISING // 外部トリガの立ち上がりで開始される
                  | ADC_EXTERNALTRIGCONV_T8_TRGO    // 外部トリガ： Timer8 TRGO event
                  | ADC_CR2_ADON;                   // ADC を有効にする
        ADC3->SQR3 = 0x1;                           // CH1 を使う
        
        // ADC 共通の設定
        ADC->CCR = 0x0;     // 念のため

        // AD 変換器の外部トリガに使うタイマ (TIM8) の設定
        SetTim8(frequency);
    }

    void AdcDual::Read(float &ad1, float &ad2)
    {
        WaitDone();
        ad1 = ToFloat(ADC2->DR);
        ad2 = ToFloat(ADC3->DR);
    }

    void AdcDual::Read(uint16_t &ad1, uint16_t &ad2)
    {
        WaitDone();
        ad1 = ADC2->DR;
        ad2 = ADC3->DR;
    }

    void AdcDual::SetTim8(int frequency)
    {
        __HAL_RCC_TIM8_CLK_ENABLE();    // Supply clock. See "stm32f4xx_hal_rcc.h"
        SystemCoreClockUpdate();        // Update core clock. See "system_stm32f4xx.h"
        TIM_TypeDef* const myTim = TIM8;

        myTim->CR2 = TIM_CR2_MMS_1; // Update event: as trigger out

    	uint32_t psc = 0;
        uint16_t mul = 1;
        uint32_t arr;
        while (true)
        {
            arr = SystemCoreClock/(mul*frequency);
            if (arr <= 65536) break;
            psc++;
            mul++;
            if (psc > 65535)
            {
                fprintf(stderr, "Sampling frequency: too low.\r\n");
                while (true) {}
            }
        }
        myTim->ARR = arr - 1;       // Auto-reload
        myTim->PSC = psc;           // Prescaler
        myTim->CR1 = TIM_CR1_CEN;   // Enable TIM8
    }
}
