//----------------------------------------------------------------------
//  NUCLEO-F446 で STM32F446 内蔵の ADC, DAC のためのクラスのデモプログラム
//      使用するクラス：DspAdcDual_Polling, DspDacDual, DspAdcDual_Intr
//      処理の内容：AD 変換器からの入力をそのまま DA 変換器に出力する
//      ポーリング方式と割り込み方式の２つの例を示す
//
//      割り込み方式では標本化周波数を PC のターミナル･ソフトで変更できる
//
//  2021/06/29, Copyright (c) 2021 MIKAMI, Naoki
//----------------------------------------------------------------------

#include "DSP_DacDual.hpp"
#include "SerialRxTxIntr.hpp"
#pragma diag_suppress 870   // マルチバイト文字使用の警告抑制のため
using namespace Mikami;

// ポーリングの例を有効にする場合，次の #define 文を無効にすること
#define ADC_EXAMPLE_OF_INTERRUPT

//----------------------------------------------------------------
// ポーリングを使う例
#ifndef ADC_EXAMPLE_OF_INTERRUPT
#include "DSP_AdcDualPolling.hpp"

int main()
{
    const float FS = 10;    // 標本化周波数: 10 kHz
    DspAdcDualPolling myAdc(FS, A0, A1);
    DspDacDual myDac;

    printf("\r\nポーリング方式\r\n");
    while (true)
    {
        // float 型の例
        float sn1, sn2;
        myAdc.Read(sn1, sn2);
        myDac.Write(sn1, sn2);
    }
}

//----------------------------------------------------------------
// ADC 変換終了割り込みを使う例
#else
#include "DSP_AdcDualIntr.hpp"

const float FS = 10;    // 標本化周波数: 10 kHz
DspAdcDualIntr myAdc_(FS, A0, A1);
DspDacDual myDac_;

// ADC 変換終了割り込みに対する割り込みサービス･ルーチン
void AdcIsr()
{
    float sn1, sn2;
    myAdc_.Read(sn1, sn2);
    myDac_.Write(sn1, sn2);
}

int main()
{
    printf("\r\n割り込み方式\r\n");
    printf("起動時の標本化周波数: 10 kHz, 標本化周波数を変更できます\r\n");
    SerialRxTxIntr rxTx;    // Serial クラスの受送信割込み用オブジェクト
    rxTx.EchobackEnable();

    myAdc_.SetIntrVec(&AdcIsr);
    myAdc_.DisableIntr();   // 一旦，AD 変換k終了割り込みを禁止する
    printf("Enter キーを押すと開始します\r\n");
    while (!rxTx.IsEol()) {}
    myAdc_.EnableIntr();    // AD 変換k終了割り込みを許可する

    rxTx.TxString("標本化周波数 [kHz] ? ");
    while (true)
    {
        if (rxTx.IsEol())
        {
            string str = rxTx.GetBuffer();
            float fS = atof(str.c_str());
            myAdc_.SetFs(fS);
            rxTx.TxString("標本化周波数 [kHz] ? ");
        }
    }
}
#endif