//-----------------------------------------------------------
//  出力を 4 倍にアップサンプリングするクラス（ヘッダ）
//  補間処理で使うフィルタ：直線位相 FIR フィルタ
//
//      出力端子： A2 (PA_4) 
//
//  2021/12/22, Copyright (c) 2021 MIKAMI, Naoki
//-----------------------------------------------------------

#include "DSP_AdcIntr.hpp"
#include "DSP_Dac.hpp"
#include "Array.hpp"

#ifndef MULTIRATE_LINEARPHASE_HPP
#define MULTIRATE_LINEARPHASE_HPP

namespace Mikami
{
    class MultirateLiPh
    {
    public:
        // コンストラクタ（デフォルトの補間フィルタの係数を使う場合）
        //      fSampling   入力の標本化周波数 [kHz]
        //      pin         入力ピン（デフォルトは A1）
        //      adc         ADC1, ADC2, ADC3 のいずれか
        explicit MultirateLiPh(float fSampling,
                               PinName pin = A1, ADC_TypeDef* const adc = ADC2);
        // コンストラクタ（デフォルト以外の補間フィルタの係数を使う場合）
        //      order       補間フィルタの次数
        //      h1, h2, h3  補間フィルタの係数
        MultirateLiPh(float fSampling, int order, const float hk1[],
                      const float hk2[], const float hk3[],
                      PinName pin = A1, ADC_TypeDef* const adc = ADC2);

        virtual ~MultirateLiPh() { delete adc_; }

        // 標本化の実行開始
        //      Func        ソフトウェア割込みに対する ISR
        void Start(void (*Func)());

        // AD変換の結果を取り出す
        float Input() const { return xn_; }

        // 補間用フィルタを実行し，処理結果を出力用バッファへ書き込む
        void Output(float yn);

    private:
        static const int UR_ = 4;   // アップサンプリング倍率：4 倍
                                    // この倍率は 2 のべき乗にすること
        static const int MASK_UR_ = UR_ - 1;
        static const int MASK_BUF_ = 2*UR_ - 1;

        static DspAdcIntr *adc_;    // AD変換器のオブジェクトのポインタ
        static DspDac dac_;         // DA変換器のオブジェクト

        static Array<float> buf_;   // DA変換器に出力するデータ用バッファ
        static int indexR_;         // buf_ から読み出す際のインデックス
        static float xn_;           // AD変換器から入力されたデータ
        int indexW_;                // buf_ へ書き込む際のインデックス

        // 補間用フィルタ用
        const int FIR_LOOP_;        // FIR フィルタのループの数
        const int CENTER_;          // 補間処理をしない信号の位置
        Array<float> vn_;           // FIR フィルタの遅延器に対応するバッファ
        Array<float> h1_, h2_, h3_; // FIR フィルタの係数

        // 補間用フィルタ用の係数（デフォルト）
        static const int ORDER_;
        static const float HK1_[], HK2_[], HK3_[];

        // 引数を 0 ～ (2*アップサンプリング倍率-1) の間でカウントアップ
        static inline int ModIndex(int &index)
        {   return ++index & MASK_BUF_; }

        // ADC 変換終了割り込みに対する割り込みサービス･ルーチン
        static void AdcIsr();

        // 補間用 FIR フィルタ
        float Interpolate(const float hk[]) const;

        // ADC の初期化と割込み優先順位の設定
        void Init(float fSampling, PinName pin, ADC_TypeDef* const adc);

        // コピー･コンストラクタ，代入演算子の禁止のため
        MultirateLiPh(const MultirateLiPh&);
        MultirateLiPh& operator=(const MultirateLiPh&);
    };
}
#endif  // MULTIRATE_LINEARPHASE_HPP