//----------------------------------------------------------
//  STM32F446 の ADC2, ADC3 を使って同時に AD 変換を開始し，
//  アナログ信号を入力するための抽象クラス（ヘッダ）
//      CH1 は ADC2 を使用
//      CH2 は ADC3 を使用
//  Read(), Write() の引数：
//      第一引数：CH1，第二引数：CH2
//
//  2020/11/07, Copyright (c) 2020 MIKAMI, Naoki
//----------------------------------------------------------

#include "mbed.h"
#include "PeripheralPins.h" // PinMap_ADC で使用
// PeripheralPins.c が含まれるディレクトリ：
//      mbed-dev\targets\TARGET_STM\TARGET_STM32F4\
//      TARGET_STM32F446xE\TARGET_NUCLEO_F446RE

#ifndef STM32F446xx
#error Select NUCLEO-F446RE.
#endif

#ifndef F446_ADC_DUAL_HPP
#define F446_ADC_DUAL_HPP

namespace Mikami
{
    class DspAdcDualBase
    {
    public:
        // コンストラクタ
        //      fSampling   標本化周波数 [kHz]
        //      pinCh1      チャンネル 1 に対応する入力ピンの名前
        //      pinCh2      チャンネル 2 に対応する入力ピンの名前
        DspAdcDualBase(float fSampling, PinName pinCh1, PinName pinCh2);

        virtual ~DspAdcDualBase() {}

        // AD 変換された値を読み込む
        //      -1.0f <= AD変換された値 < 1.0f
        virtual void Read(float &ad1, float &ad2) const = 0;    // 純粋仮想関数

        // AD 変換器の外部トリガに使うタイマ (TIM8) の設定
        //      fSampling 標本化周波数 [kHz]
        void SetFs(float fSampling);

    protected:
        ADC_TypeDef* adcCh1_;   // チャンネル 1 の AD 変換器に対応する構造体のポインタ
        ADC_TypeDef* adcCh2_;   // チャンネル 2 の AD 変換器に対応する構造体のポインタ

        float ToFloat(uint16_t x) const { return AMP_*(x - 2048); }

    private:
        static const float AMP_ = 1.0f/2048.0f;

        // AD 変換器の初期設定
        void Initialize(ADC_TypeDef* adc, PinName pin);

        //  pin に対応する AD 変換器が存在することを確認する
        bool PinmapMatch(PinName pin, ADC_TypeDef* const adc);

        // コピー･コンストラクタ，代入演算子の禁止のため
        DspAdcDualBase(const DspAdcDualBase&);
        DspAdcDualBase& operator=(const DspAdcDualBase&);
    };
}
#endif  // F446_ADC_DUAL_HPP