//--------------------------------------------------------------------
//  STM32F446 内蔵の DAC 用のクラス（ヘッダ）
//      選択可能な入力端子：
//          A2  (PA_4)： ---- デフォルト
//          D13 (PA_5)： このポートはマイコンボードの LED もドライブ
//                       するのでこのポートは使わない方がよい
//
//  2020/05/21, Copyright (c) 2020 MIKAMI, Naoki
//--------------------------------------------------------------------

#include "mbed.h"
#include "pinmap.h"         // pin_function() で使用

#ifndef DAC_BASE
#error DAC not built in.
#endif

#ifndef DSP_DAC_HPP
#define DSP_DAC_HPP

namespace Mikami
{
    class DspDac
    {
    public:
        // コンストラクタ
        explicit DspDac(PinName pin = A2);

        virtual ~DspDac() {}

        // -1.0f <= data <= 1.0f
        void Write(float data) { WriteDac(ToUint16(data)); }

        // 0 <= data <= 4095
        void Write(uint16_t data) { WriteDac(__USAT(data, BIT_WIDTH_)); }

    private:
        void (DspDac::*fpWrite_)(uint16_t);

        static const int BIT_WIDTH_ = 12;
        DAC_TypeDef* const DAC_;    // DA 変換器に対応する構造体のポインタ
        static bool created_;       // このクラスのオブジェクトの複数生成禁止で使用

        // DAC の片方のチェンネルへ出力する
        void WriteDac1(uint16_t val) { DAC->DHR12R1 = val; }    // CH1 へ
        void WriteDac2(uint16_t val) { DAC->DHR12R2 = val; }    // CH2 へ

        void WriteDac(uint16_t val) { (this->*fpWrite_)(val); }

        // 飽和処理を行い uint16_t 型のデータを戻り値とする
        uint16_t ToUint16(float val)
        {   return __USAT((val + 1.0f)*2048.0f, BIT_WIDTH_); }

        // コピー･コンストラクタ，代入演算子の禁止のため
        DspDac(const DspDac&);
        DspDac& operator=(const DspDac&);     
    };
}
#endif  // DSP_DAC_HPP