//-------------------------------------------------------------
//  STM32F446 内蔵 ADC のための抽象基底クラス(ヘッダ)
//
//  2020/12/20, Copyright (c) 2020 MIKAMI, Naoki
//-------------------------------------------------------------

#include "mbed.h"
#include "PeripheralPins.h" // PinMap_ADC で使用
// PeripheralPins.c が含まれるディレクトリ：
//      mbed-dev\targets\TARGET_STM\TARGET_STM32F4\
//      TARGET_STM32F446xE\TARGET_NUCLEO_F446RE

#ifndef DSP_ADC_BASE_HPP
#define DSP_ADC_BASE_HPP

namespace Mikami
{
    class DspAdcBase
    {
    public:
        // コンストラクタ
        //      fSampling   標本化周波数 [kHz]
        //      pin         入力ピンの名前
        //      adc         AD 変換器: ADC1, ADC2, ADC3 のいずれか
        DspAdcBase(float fSampling, PinName pin,
                    ADC_TypeDef* const adc = ADC2);

        virtual ~DspAdcBase() {}

        // AD 変換された値を読み込む
        //      -1.0f <= AD変換された値 < 1.0f
        virtual float Read() const = 0; // 純粋仮想関数

        // AD 変換器の外部トリガに使うタイマ (TIM8) の設定
        //      fSampling 標本化周波数 [kHz]
        void SetFs(float fSampling);

    protected:
        static ADC_TypeDef* myAdc_;     // AD 変換器に対応する構造体のポインタ
        float AdcToFloat() const { return AMP_*((int16_t)myAdc_->DR - 2048); }

    private:
        static const float AMP_ = 1.0f/2048.0f;
        static bool created_;   // このクラスのオブジェクトの複数生成禁止で使用

        //  pin に対応する AD 変換器が存在することを確認する
        bool PinmapMatch(PinName pin, ADC_TypeDef* const adc);

        // コピー･コンストラクタ，代入演算子の禁止のため
        DspAdcBase(const DspAdcBase&);
        DspAdcBase& operator=(const DspAdcBase&);
    };
}
#endif  // DSP_ADC_BASE_HPP