//-----------------------------------------------------------------
//  ディジタルフィルタ実験のためのプラットフォーム（標本化周波数：48 kHz）
//      ---- 入出力信号の波形表示付き ----
//
//  適応線スペクトル強調器（ALE）：
//          ターミナルソフトからステップサイズパラメータ（μ）の変更可能
//
//  使用しているライブラリのリビジョン：
//          F746_GUI            Rev.33
//          F746_SAI_IO         Rev.12
//          mbed                Rev.161
//
//  2018/09/04, Copyright (c) 2018 MIKAMI, Naoki
//-----------------------------------------------------------------

#include "SAI_InOut.hpp"
#include "LcdPanelFrqEx.hpp"
#include "TerminalInput.hpp"    // ターミナルとの通信用
#include "GaussRand.hpp"        // 正規乱数発生

#pragma diag_suppress 870       // マルチバイト文字使用の warning 抑制のため

//-----------------------------------------------------------------
// 実行するフィルタに応じて以下のインクルードファイルを変更する
#include "ALE_Variable.hpp"    // 線スペクトル強調器
//-----------------------------------------------------------------

int main()
{
    //-------------------------------------------------------------
    // xxxx_Variable 版の追加箇所
    const string MESSAGE = "1.0E-10 以下の数を入力してください\r\n";
    TermInput comm("\r\n適応線スペクトル強調器を実行します．\r\n");
    comm.printf("ここからμ (ステップサイズパラメータ) の値を変更できます．\r\n");
    comm.printf(MESSAGE+"? ");
    // xxxx_Variable 版の追加箇所：ここまで
    //-------------------------------------------------------------

    const int FS = I2S_AUDIOFREQ_48K;   // 標本化周波数：48 kHz
    // オーディオ信号出力用
    SaiIO mySai(SaiIO::OUTPUT, 2048, FS);

    // LCD パネル表示用
    LcdPanelEx myPanel(1.0f/(float)FS, 1);

    //-----------------------------------------------------------------
    // ここでディジタルフィルタのオブジェクト生成や初期化などを行う
    float mu = 1.0E-12f;
    AleVariable df(5, 120, mu);         // ALE を実行するオブジェクト
    Mikami::GaussRand rd(0.6f, 0);      // ガウス性白色雑音，標準偏差：0.6，平均値： 0
    //-----------------------------------------------------------------
    
    // ステップサイズ･パラメータの表示用ラベル
    NumericLabel<float> myLabel(40, 168, "mu = %8.2e");
    myLabel.Draw(mu);
    
    mySai.PlayOut();    // オーディオ信号出力開始

    const float MAX_AMP = 10000.0f;
    string rxBuf = "";
    while (true)        // 無限ループ
    {
        // 出力バッファへの転送が完了したら以下の処理を行う
        if (mySai.IsXferred())
        {
            for (int n=0; n<mySai.GetLength(); n++)
            {
                // 入力信号の生成
                float phi = myPanel.GetPhi();
                float sn = sinf(phi);       // 入力信号となる正弦波を発生
                sn = 0.6f*(sn + rd.Next()); // 雑音を付加

                int16_t xn = (int16_t)(MAX_AMP*sn);
                //------------------------------------------------------
                // ディジタルフィルタなどの信号処理の実行
                int16_t yn = df.Execute(xn);
                //------------------------------------------------------

                // ヘッドフォン端子への出力信号と表示する信号の選択
                switch (myPanel.GetOutNum())
                {
                    case 0: mySai.Output(xn, yn);       // 入出力信号
                            myPanel.Store(xn, yn, n);   // 表示用として格納
                            break;
                    case 1: mySai.Output(xn, xn);       // 両チャンネル：入力信号
                            myPanel.Store(xn, 0, n);    // 表示用として格納
                            break;                       
                    case 2: mySai.Output(yn, yn);       // 両チャンネル：出力信号
                            myPanel.Store(0, yn, n);    // 表示用として格納
                            break;
                }
            }

            myPanel.Display(); // 入出力信号の波形表示
        }
        myPanel.Update();   // パネルの状態を読み取り必要なパラメータなどを更新する

        //---------------------------------------------------------
        // xxxx_Variable 版の追加箇所
        // ターミナルとの通信
        float val;
        if (comm.Get(val))  // 入力が完了していれば以下を実行する
        {
            if (val <= 1.0E-10f)
                df.SetMu(val);  // μ の変更
            else
                comm.printf(MESSAGE);

            myLabel.Draw(val);
            comm.printf("? ");
        }
        // xxxx_Variable 版の追加箇所：ここまで
        //---------------------------------------------------------
    }
}
