//-------------------------------------------------------------------
//  ディジタルフィルタ実験のためのプラットフォームの
//  LCD パネル表示と入力信号の周波数設定で使う
//
//  "LcdPanelFrq.hpp" の LcdPanel に対して以下の変更および追加を行った
//      変更： トリガのソースを入力／出力信号に選択できるようにした
//      追加： ヘッドフォン出力の切り替えボタンを追加した
//
//  2018/03/02, Copyright (c) 2018 MIKAMI, Naoki
//-------------------------------------------------------------------

#ifndef PANEL_MANIPULATOR_HPP
#define PANEL_MANIPULATOR_HPP

#include "F746_GUI.hpp"
#include "NumericUpDown.hpp"
#include "Zoom.hpp"
#include "WaveformDisplay.hpp"
#include "Trigger.hpp"

namespace Mikami
{
    class LcdPanelEx
    {
    public:
        //  ts  : 標本化間隔（単位：μs）
        //  io  : トリガのソースの選択
        //          io == 0 : 入力信号（デフォルト）
        //          io != 0 : 出力信号
        LcdPanelEx(float ts, int io = 0)
            : TS_(ts), lcd_(GuiBase::GetLcd()),
              xnD_(N_DISP_BUF_), ynD_(N_DISP_BUF_), sn_((io==0)? xnD_: ynD_),
              frq_(1000), dPhi_(PI2_*frq_*TS_), phi_(0), factor_(1), touchedNum_(0)
        {
            Label title(240, 10, "Digital Filter LAB", Label::CENTER, Font16);

            const uint16_t Y0 = 170;
            // 凡例（ヘッドフォン端子への出力の切り替え用の ButtonGroup を
            //      追加したので他のアプリ用とは違っている）
            lcd_.SetTextColor(LINE_COLOR1_);
            lcd_.DrawHLine(396, Y0+6, 30);
            Label label1(430, Y0, "Input");
            lcd_.SetTextColor(LINE_COLOR2_);
            lcd_.DrawHLine(396, Y0+26, 30);
            Label label2(430, Y0+20, "Output");

            waveDisp_ = new WaveformDisplay(lcd_, 30, 90, xnD_, ynD_, 400, 8,
                                            AXIS_COLOR_,
                                            LINE_COLOR1_, LINE_COLOR2_,
                                            BACK_COLOR_);

            // GUI 設定
            ud_ = new NumericUpDown(4, 0, 271, frq_, 9999, 100, "Hz");
            bar_ = new SeekBar(230, 250, 230,
                               log10f(MIN_FRQ_), log10f(MAX_FRQ_), log10f(1000),
                               "0.1", "1.0", "10.0");
            // 時間軸の拡大率は 8 倍まで
            zmButton_ = new ZoomButton(180, Y0, 36, 30, 8);
            
            // ヘッドフォン端子への出力の切り替え（処理前と処理後の信号の切り替え）
            // 追加したもの
            outSw_ = new ButtonGroup(270, Y0, 36, 30, 3,
                                     (string[]){"IO", "I", "O"}, 3, 0, 3);
            outSw_->TouchedColor(0);
        }

        // 正弦波の発生で使う値の更新し，取得する
        float GetPhi()
        {
            if ((phi_ += dPhi_) > PI2_) phi_ -= PI2_;
            return phi_;
        }

        // 周波数設定の更新とズームイン／アウト･ボタンの処理
        void Update()
        {        
            if (bar_->Slide())  // SeekBar をスライドした場合の処理
            {
                frq_ = (int)(powf(10, bar_->GetValue())+0.5f);
                if (frq_ >= MAX_FRQ_) frq_ = MAX_FRQ_ - 1;
                ud_->Set(frq_); // SeekBar の数値を NumericUpDown に表示する
                dPhi_ = PI2_*frq_*TS_;
            }

            if (ud_->Touched())     // NumericUpDown をタッチした場合の処理
            {
                frq_ = ud_->Get();
                bar_->Draw(log10f(frq_));   // NumericUpDown の数値で SeekBar のツマミの位置を設定
                dPhi_ = PI2_*frq_*TS_;
            }

            zmButton_->Touched(factor_);    // ズームインとズームアウト
            
            outSw_->GetTouchedNumber(touchedNum_);  // タッチした番号を取得
        }

        // 入出力信号のデータを表示用バッファへ格納
        void Store(int16_t xn, int16_t yn, int n)
        {
            if (n >= N_DISP_BUF_) return;

            xnD_[n] = xn;
            ynD_[n] = yn;
        }
        
        // 入出力信号の波形の表示
        void Display() { waveDisp_->Execute(Trigger(sn_, 1), factor_); }
        
        // 出力切り替え用 ButtonGroup のタッチされた番号を取得
        int GetOutNum() { return touchedNum_; }

    private:
        static const float PI_ = 3.1415926536f;
        static const float PI2_ = 2.0f*PI_;
        static const int MIN_FRQ_ = 100;        // 発生する周波数の最小値
        static const int MAX_FRQ_ = 10000;      // 発生する周波数の最大値
        static const uint32_t BACK_COLOR_ = GuiBase::ENUM_BACK;
        static const uint32_t AXIS_COLOR_ = 0xFFCCFFFF;
        static const uint32_t LINE_COLOR1_ = LCD_COLOR_CYAN;
        static const uint32_t LINE_COLOR2_ = LCD_COLOR_MAGENTA;
        static const int N_DISP_BUF_ = 1000;    // 波形表示用のバッファのサイズ

        const float TS_;        // 標本化間隔

        LCD_DISCO_F746NG &lcd_;
        WaveformDisplay *waveDisp_;
        NumericUpDown *ud_;     // 入力信号の周波数設定用（1 Hz 単位での設定用）
        SeekBar *bar_;          // 入力信号の周波数設定用（概略の設定用）
        ZoomButton *zmButton_;  // 波形表示：ズームイン/アウト用
        ButtonGroup *outSw_;    // 出力の切り替え

        Array<int16_t> xnD_;    // 波形表示で使うバッファ（入力信号用）
        Array<int16_t> ynD_;    // 波形表示で使うバッファ（出力信号用）
        const Array<int16_t> &sn_;  // Trigger() の引数に渡す参照
        int frq_;               // 現在の入力信号の周波数，初期値は 1 kHz
        float dPhi_, phi_;      // 入力信号の生成で使う変数
        int factor_;            // 波形表示の際の標本化間隔に対応するピクセル数
        int touchedNum_;        // 出力選択ボタンの番号
    };
}
#endif  // PANEL_MANIPULATOR_HPP
