//-----------------------------------------------------------
//  ディジタルフィルタ実験のためのプラットフォーム
//  信号処理の対象とする信号は Arduino 互換ヘッダピンの A0 に入力する
//      ---- 入出力信号の波形表示付き ----
//
//  このプログラムでは，例として定数乗算器を実装している
//
//  使用しているライブラリのリビジョン：
//          F746_GUI            Rev.33
//          F746_SAI_IO         Rev.12
//          mbed                Rev.161
//
//  2018/03/12, Copyright (c) 2018 MIKAMI, Naoki
//-----------------------------------------------------------

#include "SAI_InOut.hpp"
#include "LcdPanel.hpp"
#include "MyFunctions.hpp"  // ST32F746 内蔵の ADC を使うため

//-----------------------------------------------------------------
// 実行するフィルタの応じて以下のインクルードファイルを変更する
#include "Multiplier.hpp"   // 入力を定数倍して出力する処理
//-----------------------------------------------------------------

const int FS_ = I2S_AUDIOFREQ_48K;  // 標本化周波数：48 kHz
AdcDual_Intr adc_(FS_);             // 内蔵 ADC 用

int main()
{
    // オーディオ信号出力用
    SaiIO mySai(SaiIO::OUTPUT, 2048, FS_);

    // LCD パネル表示用
    LcdPanel myPanel;

    //-----------------------------------------------------------------
    // ここでディジタルフィルタのオブジェクト生成や初期化などを行う
    Multiplier df(-0.5f);  // 定数乗算の処理
    //-----------------------------------------------------------------

    // 内蔵 ADC を使うための初期化
    AdcInitialize(mySai.GetLength());

    mySai.PlayOut();    // オーディオ信号出力開始
    while (true)        // 無限ループ
    {
        // 出力バッファへの転送完了と ADC バッファが満杯で以下の処理を行う
        if (mySai.IsXferred() && IsAdcBufferFull())
        {
            for (int n=0; n<mySai.GetLength(); n++)
            {
                int16_t sn, dummy;
                GetSignal(sn, dummy);   // A0 から入力された信号を取得する

                int16_t xn = sn << 3;
                //------------------------------------------------------
                // ディジタルフィルタなどの信号処理の実行
                int16_t yn = df.Execute(xn);    // ディジタルフィルタの処理
                //------------------------------------------------------
                mySai.Output(xn, yn);       // xn：入力（左チャンネル）
                                            // yn：出力（右チャンネル）
                myPanel.Store(xn, yn, n);   // 表示用として格納
            }
            myPanel.Display();  // 入出力信号の波形表示
        }
        myPanel.Update();
    }
}
