//----------------------------------------------------------
//  Simultanuous AD Conversion by polling using
//  ADC1 and ADC3 on STM32F746 ---- Header
//
//  STM32F746 の ADC1, ADC3 を使って同時に AD 変換を開始し，
//  ポーリングによりアナログ信号を入力するクラス
//      PA_0  (DISCO-F746 の A0) :  ADC1 CH0
//      PF_10 (DISCO-F746 の A1) :  ADC3 CH8
//
//  Read() の引数：
//      第一引数：A0 (左)，第二引数：A1 (右)
//
//  2017/08/16, Copyright (c) 2017 MIKAMI, Naoki
//----------------------------------------------------------

#include "mbed.h"

#ifndef STM32F746xx
#error Target is not STM32F746
#endif

#ifndef F746_ADC_DUAL_HPP
#define F746_ADC_DUAL_HPP

namespace Mikami
{
    class AdcDual
    {
    public:
        // Constructor
        //      frequency: 標本化周波数
        explicit AdcDual(int frequency);
        
        virtual ~AdcDual() {}

        // -1.0f <= ad1, ad2 <= 1.0f
        //      ad1: left, ad2: right
        virtual void Read(float &ad1, float &ad2);

        // 0 <= ad1, ad2 <= 4095
        //      ad1: left, ad2: right
        virtual void Read(uint16_t &ad1, uint16_t &ad2);

    protected:
        float ToFloat(uint16_t x)
        {   return AMP_*(x - 2048); }
    
    private:
        static const float AMP_ = 1.0f/2048.0f;
        static const uint32_t EOC13_ = ADC_CSR_EOC1 | ADC_CSR_EOC3;
        
        // AD 変換が完了するまで待つ
        void WaitDone()
        {   while((ADC->CSR & EOC13_) != EOC13_); }

        // AD 変換器の外部トリガに使うタイマ (TIM6) の設定
        void SetTim6(int frequency);

        // for inhibition of copy constructor
        AdcDual(const AdcDual&);
        // for inhibition of substitute operator
        AdcDual& operator=(const AdcDual&);     
    };
}
#endif  // F746_ADC_DUAL_HPP

/*
typedef struct
{
  __IO uint32_t SR;     //!< ADC status register,                         Address offset: 0x00 //
  __IO uint32_t CR1;    //!< ADC control register 1,                      Address offset: 0x04 //      
  __IO uint32_t CR2;    //!< ADC control register 2,                      Address offset: 0x08 //
  __IO uint32_t SMPR1;  //!< ADC sample time register 1,                  Address offset: 0x0C //
  __IO uint32_t SMPR2;  //!< ADC sample time register 2,                  Address offset: 0x10 //
  __IO uint32_t JOFR1;  //!< ADC injected channel data offset register 1, Address offset: 0x14 //
  __IO uint32_t JOFR2;  //!< ADC injected channel data offset register 2, Address offset: 0x18 //
  __IO uint32_t JOFR3;  //!< ADC injected channel data offset register 3, Address offset: 0x1C //
  __IO uint32_t JOFR4;  //!< ADC injected channel data offset register 4, Address offset: 0x20 //
  __IO uint32_t HTR;    //!< ADC watchdog higher threshold register,      Address offset: 0x24 //
  __IO uint32_t LTR;    //!< ADC watchdog lower threshold register,       Address offset: 0x28 //
  __IO uint32_t SQR1;   //!< ADC regular sequence register 1,             Address offset: 0x2C //
  __IO uint32_t SQR2;   //!< ADC regular sequence register 2,             Address offset: 0x30 //
  __IO uint32_t SQR3;   //!< ADC regular sequence register 3,             Address offset: 0x34 //
  __IO uint32_t JSQR;   //!< ADC injected sequence register,              Address offset: 0x38 //
  __IO uint32_t JDR1;   //!< ADC injected data register 1,                Address offset: 0x3C //
  __IO uint32_t JDR2;   //!< ADC injected data register 2,                Address offset: 0x40 //
  __IO uint32_t JDR3;   //!< ADC injected data register 3,                Address offset: 0x44 //
  __IO uint32_t JDR4;   //!< ADC injected data register 4,                Address offset: 0x48 //
  __IO uint32_t DR;     //!< ADC regular data register,                   Address offset: 0x4C //
} ADC_TypeDef;

typedef struct
{
  __IO uint32_t CSR;    //!< ADC Common status register,                  Address offset: ADC1 base address + 0x300 //
  __IO uint32_t CCR;    //!< ADC common control register,                 Address offset: ADC1 base address + 0x304 //
  __IO uint32_t CDR;    //!< ADC common regular data register for dual
                             AND triple modes,                            Address offset: ADC1 base address + 0x308 //
} ADC_Common_TypeDef;
*/
