//-----------------------------------------------------
//  内蔵 ADC を使うための関数等
//
//  2017/10/07, Copyright (c) 2017 MIKAMI, Naoki
//-----------------------------------------------------

#include "MyFunctions.hpp"
using namespace Mikami;

extern AdcDual_Intr adc_;   // 内蔵 ADC 用（割り込みを使用）

Array<int16_t> buffer1L_, buffer1R_, buffer2L_, buffer2R_;

int32_t nData_;                 // バッファサイズ
int16_t *inBufL_, *inBufR_;     // 現在使用中の入力バッファに対応
int16_t *outBufL_, *outBufR_;   // 現在使用中の出力バッファに対応
int     countIn_, countOut_;    // 入力データ数のカウント用
int     indexIn_, indexOut_;    // バッファの切り替え用
int16_t *pingPongL_[2], *pingPongR_[2]; // ２組のバッファに対応するポインタを格納

__IO bool fullIn_;              // 入力バッファが満杯のとき true

// バッファの切り替えで使うインデックスの変更
void SwitchBuf(int &index) { index = ++index & 0x01; }

// ADC のデータをブロック単位で扱うための初期化
void AdcInitialize(int32_t size)
{
    nData_ = size;

    buffer1L_.SetSize(nData_);
    buffer1R_.SetSize(nData_);
    buffer2L_.SetSize(nData_);
    buffer2R_.SetSize(nData_);

    countIn_ = countOut_ = 0;
    indexIn_ = indexOut_ = 0;    // バッファの切り替え用

    pingPongL_[0] = buffer1L_;
    pingPongL_[1] = buffer2L_;
    pingPongR_[0] = buffer1R_;
    pingPongR_[1] = buffer2R_;
            
    inBufL_ = pingPongL_[indexIn_];
    inBufR_ = pingPongR_[indexIn_];
    outBufL_ = pingPongL_[indexOut_];
    outBufR_ = pingPongR_[indexOut_];

    fullIn_ = false;
    adc_.SetIntrVec(&AdcIsr);   // ADC の割り込みベクタに ISR を割り当て
    adc_.EnableAdcIntr();       // ADC 終了割込みを有効にする
}

// ADC 終了割り込みに対する ISR (割り込みサービスルーチン)
void AdcIsr()
{
    uint16_t sn1, sn2;

    adc_.Read(sn1, sn2);
    inBufL_[countIn_] = sn1 - 2048;
    inBufR_[countIn_++] = sn2 - 2048;

    if (countIn_ == nData_)
    {
        fullIn_ = true;
        countIn_ = 0;           // 入力バッファの先頭を指す
        SwitchBuf(indexIn_);    // 入力バッファのインデックスを切り替える
        inBufL_ = pingPongL_[indexIn_];
        inBufR_ = pingPongR_[indexIn_];
    }
}

// 入力バッファが満杯かどうか調べる，満杯であれば true を返す
bool IsAdcBufferFull()
{
    if (!fullIn_) return false;

    fullIn_ = false;
    return true;
}

// AD 変換されたデータを取り出す
void GetSignal(int16_t &sn1, int16_t &sn2)
{
    sn1 = outBufL_[countOut_];
    sn2 = outBufR_[countOut_++];

    if (countOut_ == nData_)
    {
        countOut_ = 0;
        SwitchBuf(indexOut_);
        outBufL_ = pingPongL_[indexOut_];
        outBufR_ = pingPongR_[indexOut_];
    }
}
