//-----------------------------------------------------------------
//  ディジタルフィルタ実験のためのプラットフォーム（標本化周波数：48 kHz）
//      ---- 入出力信号の波形表示付き ----
//
//  このプログラムでは，例として定数乗算器を実装している
//
//  使用しているライブラリのリビジョン：
//          F746_GUI            Rev.33
//          F746_SAI_IO         Rev.12
//          mbed                Rev.161
//
//  2018/03/20, Copyright (c) 2018 MIKAMI, Naoki
//-----------------------------------------------------------------

#include "SAI_InOut.hpp"
#include "LcdPanelFrq.hpp"

//-----------------------------------------------------------------
// 実行するフィルタに応じて以下のインクルードファイルを変更する
#include "Multiplier.hpp"       // 入力を定数倍して出力する処理
//-----------------------------------------------------------------

int main()
{
    const int FS = I2S_AUDIOFREQ_48K;   // 標本化周波数：48 kHz
    // オーディオ信号出力用
    SaiIO mySai(SaiIO::OUTPUT, 2048, FS);

    // LCD パネル表示用
    LcdPanel myPanel(1.0f/(float)FS);

    //-----------------------------------------------------------------
    // ここでディジタルフィルタのオブジェクト生成や初期化などを行う
    Multiplier df(-0.5f);  // 定数乗算の処理
    //-----------------------------------------------------------------
    
    mySai.PlayOut();    // オーディオ信号出力開始

    const float MAX_AMP = 10000.0f;
    while (true)        // 無限ループ
    {
        // 出力バッファへの転送が完了したら以下の処理を行う
        if (mySai.IsXferred())
        {
            for (int n=0; n<mySai.GetLength(); n++)
            {
                // 入力信号の生成
                float phi = myPanel.GetPhi();
                float sn = sinf(phi); // 入力信号となる正弦波を発生

                int16_t xn = (int16_t)(MAX_AMP*sn);
                //------------------------------------------------------
                // ディジタルフィルタなどの信号処理の実行
                int16_t yn = df.Execute(xn);
                //------------------------------------------------------
                mySai.Output(xn, yn);       // xn：入力（左チャンネル）
                                            // yn：出力（右チャンネル）
                myPanel.Store(xn, yn, n);   // 表示用として格納
            }

            myPanel.Display();  // 入出力信号の波形表示
        }
        myPanel.Update();   // パネルの状態を読み取り必要なパラメータなどを更新する
    }
}
