//------------------------------------------------------
//  Class for LCD, ACM1602NI (Header)
//
//  Default pin assignments
//      D14  SDA ---- to pin5 of LCD module
//      D15  SCL ---- to pin4 of LCD module
//
//  Assignment of I2C ports
//                SDA                SCL
//      I2C1    PB_7 or PB_9(D14)   PB_6(D10) or PB_8(D15)
//      I2C2    PB_3(D3)            PB_10(D6)
//      I2C3    PB_4(D5) or PC_9    PA_8(D7)
//
//  2014/08/27, Copyright (c) 2014 MIKAMI, Naoki
//------------------------------------------------------

#ifndef ACM1602NI_HPP
#define ACM1602NI_HPP

#include "mbed.h"

namespace Mikami
{
    class Acm1602Ni
    {
    public:
        // Constructor
        Acm1602Ni(PinName sda = D14,        // SDA
                  PinName scl = D15,        // SCL
                  uint32_t clock = 100000,  // clock: 100 kHz
                  bool cursor = false,      // cursor:  off
                  bool blink = false);      // blink:   off
        // Return false if LCD is not connected
        bool GetOk() { return connected_; }
        // All clear
        bool Clear();
        // Send command
        bool WriteCmd(uint8_t cmd) { return LcdTx(0x00, cmd); }
        // Write character
        void WriteChar(char data) { LcdTx(0x80, data); }
        // Specify display position, x: 0 - 15, y: 0, 1
        void SetXY(uint8_t x = 0, uint8_t y = 0) { WriteCmd(x + y*0x40 | 0x80);}
        // Write string from specified position
        void WriteString(const char str[], uint8_t x = 0, uint8_t y = 0);
        // Clear of specified line
        void ClearLine(uint8_t line)
        { WriteString("                 ", 0, line); }

    private:
        // Slave address of ACM1602NI (0x50)
        static const uint8_t LCD_ADDRESS_ = 0x50 << 1;  // left-justified 7-bit address

        bool connected_;    // Set false in constructor if LCD is not connected

        bool LcdTx(uint8_t cmdData, uint8_t data);
        bool Start();

        // Forbid to use copy constructor
        Acm1602Ni(const Acm1602Ni&);
        // Forbid to use substitution operator
        Acm1602Ni& operator=(const Acm1602Ni&);

        uint32_t* i2c_cr1_;   // pointer to I2C_CR1 (control register 1)
        uint32_t* i2c_dr_;    // pointer to I2C_DR (data register)
        uint32_t* i2c_sr1_;   // pointer to I2C_SR1 (status register 1)
        uint32_t* i2c_sr2_;   // pointer to I2C_SR2 (status register 2)
        I2C i2c_;   // Object of I2C

        // For check contents of register
        bool Check(uint32_t* rg, uint16_t value)
        { return value == ((*rg) & value); }

        void TxDR(uint8_t data) { *i2c_dr_ = data; }
        void SetCR1(uint16_t data) { *i2c_cr1_ |= data; }
    };
}
#endif  // ACM1602NI_HPP
