#ifndef __MCP7940_H__
#define __MCP7940_H__

#include "mbed.h"
/** MCP7940 class.
 *  Interface to the MCP7940 allowing you to get and set the time
 * Example:
 * @code
 * #include "mbed.h"
 * #include "MCP7940.h"
 *
 * MCP7940 clock(PinName sda, PinName scl);
 *
 * int main() {
 *
 *      clock.setTime(2015, 9, 26, 22, 50);
 *      clock.StartClock();
 *      printf("%s \r\n" clock.TimeStamp());
 *      //or
 *      clock.getTime();
 *      printf("Day: %i, Month: %i, Year %i  \r\n" clock.Day(), clock.Month(), clock.Year());
 * }
 * @endcode
 */

class MCP7940
{
public:
    /** Constructor
             * Start year will default to 2000.
             * @param sda I2C SDA pin
             * @param scl I2C SCL pin
             */
    MCP7940(PinName sda, PinName scl);
    /** Constructor
             * Start year will default to 2000.
             * @param sda I2C SDA pin
             * @param scl I2C SCL pin
             * @param StarYear The start year for you calender. The MCP7940 defines years as XX, so that this library can give XXXX years you need to set your year 0, ie 2000
             */
    MCP7940(PinName sda, PinName scl, int StartYear);
    MCP7940();
    /** Day
    * Note getTime() must have been called to update the value of day.
    * @returns The day of the month
    */
    int Day();
    /** Month
    * Note getTime() must have been called to update the value of day.
    * @returns The month.
    */
    int Month();
    /** Year
    * Note getTime() must have been called to update the value of day.
    * @returns The Year as XXXX = StartYear + XX as in MCP7940
    */
    int Year();
    /** Hour
    * Note getTime() must have been called to update the value of day.
    * @returns the hour
    */
    int Hour();
    /** Minutes
    * Note getTime() must have been called to update the value of day.
    * @returns Minutes
    */
    int Minutes();
    /** Seconds
    * Note getTime() must have been called to update the value of day.
    * @returns seconds
    */
    int Seconds();
    /** MilliSeconds
    * Note getTime() must have been called to update the value of day.
    * Included for completeness - but MCP7940 doesn't have a mS counter
    * @returns Milliseconds = 0 for MCP7940
    */
    int MilliSeconds();
    /** DayOfWeek
    * Note getTime() must have been called to update the value of day.
    * @returns The day of the week, a number 1-7. This is user defined
    */
    int DayOfWeek();

    /** SetTime
    * Transfers the values of the time members into the MCP7940. Use one of the other setTime functions
    */
    int setTime();
    /** SetTime
    * Transfers the values of the time members into the MCP7940. Use one of the other setTime functions
    * @param Year the year in YYYY
    * @param Month
    * @param Day the Day of the month
    * @param Hour
    * @param Minutes
    */
    int setTime(int Year, int Month, int Day, int Hour, int Mins);
    /** SetTime
    * Transfers the values of the time members into the MCP7940. Use one of the other setTime functions
    * @param Year the year in YYYY
    * @param Month
    * @param Day the Day of the month
    * @param Hour
    * @param Minutes
    * @param Secs
    */
    int setTime(int Year, int Month, int Day, int Hour, int Mins, int Secs);
    /** SetTime
    * Transfers the values of the time members into the MCP7940. Use one of the other setTime functions
    * @param Year the year in YYYY
    * @param Month
    * @param Day the Day of the month
    * @param Hour
    * @param Minutes
    * @param Secs
    * @param MiliSecs  - note only for completeness and future extension as MCP7940 doesn't include mS
    */
    int setTime(int Year, int Month, int Day, int Hour, int Mins, int Secs, int MiliSecs);
    /** SetTime
    * Transfers the values of the time members into the MCP7940. Use one of the other setTime functions
    * @param Year the year in YYYY
    * @param Month
    * @param Day the Day of the month
    * @param Hour
    * @param Minutes
    * @param Secs
    * @param MiliSecs  - note only for completeness and future extension as MCP7940 doesn't include mS
    * @param DayOfWeek  - the day of the week as a number 1-7, User defined
    */
    int setTime(int Year, int Month, int Day, int Hour, int Mins, int Secs, int MiliSecs, int DayOfWeek);

    int setDefaultTime();
    /** getTime
    * Loads the time from the MCP7940 so its upto date and can be accessed by the Day(), Month()... functions
    */
    int getTime();
    /** StartClock
    * Clears and sets bit 7 of register 0x00 to start the clock. Keeps the current values of 0x00 bits 0-6
    */
    void StartClock();

    /** TimeStamp
        * getTime() is called within this function so does not need to be called from code.
        * @returns A pointer to a time stamp as YYYY-MM-DDTHH:MM:SS
        */
    char * TimeStamp();
    void TimeStamp(char * buf);
    void niceTimeStamp(char * buf);
    void niceDate(char * buf);
    void niceTime(char * buf);

private:
    char _TimeStamp[20];
    int _Day;
    int _Month;
    int _Year;
    int _Hour;
    int _Minutes;
    int _Seconds;
    int _MilliSeconds;
    int _DayOfWeek;

    int _YearStart;

protected:
    I2C Clock;
    char IntToBCD(char Data);
    char BCDtoInt(char Data);
    int _addr;


};


#endif