/**
* @section LICENSE
*Copyright (c) 2010 ARM Ltd.
*
*Permission is hereby granted, free of charge, to any person obtaining a copy
*of this software and associated documentation files (the "Software"), to deal
*in the Software without restriction, including without limitation the rights
*to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
*copies of the Software, and to permit persons to whom the Software is
*furnished to do so, subject to the following conditions:
* 
*The above copyright notice and this permission notice shall be included in
*all copies or substantial portions of the Software.
* 
*THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
*IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
*FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
*AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
*LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
*OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
*THE SOFTWARE.
* 
*
* @section DESCRIPTION
* Library for using the SCP1000-D01 MEMS Pressure sensor, this is the SPI version. This library only supports high resolution mode.
* Communication with the sensor is via a 4 wire interface
*
*/

#include "scp1000.h"

SCP1000::SCP1000(PinName mosi, PinName miso, PinName sck, PinName CSB):
     _spi(mosi, miso, sck),
    _CSB(CSB)
    {
    _CSB = 1;
    wait(0.1);
    _CSB = 0;
    //Force reset
    _spi.write(((0x06<< 2) | 0x02));
    _spi.write(0x01);
    _CSB = 1;
    wait(0.06);
    _CSB = 0;
    //Check starup Procedure has finished
    int status;
     do{
        _spi.write(0x07 << 2);
        status = _spi.write(0x00);
        //printf("waiting for startup to finish %i\n", status);
        wait(0.1);
    }while((status & 0x01));                                  //Wait for LSB to go low
    //Test for error in intialisation
    _spi.write(0x1F << 2);
    status = _spi.write(0x00);
    if(!(status & 0x01)){
        //printf("Error in Intialisation");
        return;
    }
    
    
    //Set mode as 0x00
    _spi.write((0x03 << 2) | 0x02);
    _spi.write(0x00);
    wait(0.05);
    
    //Check DRDDY is low, if not read data
    _spi.write(0x07 << 2);
    status = _spi.write(0x00);
    if(status & 0x20){
       //printf("Data to be read");
       _spi.write(0x1F <<2);
       _spi.write(0x00);        //read and discard data
       _spi.write(0x20 <<2);
       _spi.write(0x00);       //read and discard data
       _spi.write(0x00);      //read and discard data
    }
    
    //Check OPStatus bit
    _spi.write(0x04 << 2);
    status = _spi.write(0x0);
    if(status & 0x01){
        //printf("Not finished");
    }   
    
    //Activate new mode
    _spi.write((0x03 << 2) | 0x02);
    _spi.write(0x0A);
}
float SCP1000::read(){
    _CSB = 0;

    if(_waitReady() == 1){ 
        _spi.write(0x1F <<2);
        int PressureHighest = _spi.write(0x00);
        _spi.write(0x20 <<2);
        int PressureHigh = _spi.write(0x00);
        int Pressurelow = _spi.write(0x00);
        
        int pressureValue = Pressurelow | PressureHigh << 8 | (PressureHighest & 0x07) << 16;       
        float pressure = ((float)pressureValue) / 4;
 
        _CSB = 1;

        return(pressure);
    }else{
        return(0);
    }
}
float SCP1000::readTemperature(){
    
    _CSB = 0;
    
    if(_waitReady() == 1){ 
        //ready so now read
         _spi.write(0x21 << 2);
        int TempHigh = _spi.write(0x00);
        int TempLow = _spi.write(0x00);
        
        signed int temperatureValue = (TempLow | ((TempHigh & 0x1F) << 8));
        if(TempHigh & 0x20){
            //negative
            temperatureValue = -8192 + temperatureValue;
        }else{
            //positive         
        }

        float temperature = ((float)temperatureValue) * 0.05;
        _CSB = 1;
        return(temperature);
    }else{
        return(0);
    }
}
    
 int SCP1000::_waitReady(){
    //Depending on mode wait for it to be ready - only supports high resolution mode - wait for bit 5 to be set in 00
    int status;
    _CSB = 0;
    do{
        _spi.write(0x07 << 2);
        status = _spi.write(0x00);
        //printf("waiting %i\n", status);
        wait(0.2);
        if(status & 0x10){
            //bit 4 high - real time error, interrupt has not been read in time - read DataRD16
            _spi.write(0x20 << 2);
            int data = _spi.write(0x00);
            data = _spi.write(0x00);
        }
    }while(!(status & 0x20));
    return(1);
 }