#ifndef L298HBridge_H
#define L298HBridge_H
 
#include "mbed.h"
#include <string>
#include <vector>
using namespace std;
 
/** Class library for a L298 H-Bridge.
 *
 * Example:
 * @code
 * #include "mbed.h"
 * #include "L298HBridge.h"
 *
 * L298HBridge dcmotor(PA_5, PA_6, PA_10); //Create a L298HBridge object with enable pin on PA5, Fwd pin or PA6 and Rev pin on PA10
 *
 * int main() 
 * {
 *    while(1)
 *    {
 *          dcmotor.Fwd();               //setting the motor to spin forward
 *          dcmotor.SetSpeed(0.5);       //setting the speed the motor will spin
 *    }    
 * }
 * @endcode
 */
 
class L298HBridge {
  public:
    /** Create a L298HBridge object connected to the specified pins. Once created, The pins still need
    * to be set to a direction and also the speed via PWM.
    * @param ENpin PwmOut compatible pin used to set the speed of the motor.
    * @param FWDpin Used to spin the motor in the forward direction.
    * @param REVDpin Used to spin the motor in the Reverse direction.
    * @param chanA digital interrupt to trigger the speed calculation.
    * @param chanB digital input to get the sens of rotation.
    */
    L298HBridge(PinName ENpin, PinName FWDpin, PinName REVpin, PinName chanApin, PinName chanBpin);
    
    /** Setting the DC motor to spin in the forward direction.
    * @param 
    *     None
    */
    void Fwd();
    
    /** Setting the DC motor to spin in the Revers direction.
    * @param 
    *     None
    */
    void Rev();
    
     /** Stopping the motor.
    * @param 
    *     None
    */
    void Stop();
    
     /** Set the speed of the motor.
    * @param 
    *     PWMPercentage, variable to set te speed of the motor (any value from 0.1 - 1).
    * @return
    *     None
    */
    void SetSpeed(float PWMPercentage);
    
    /** Get the speed of the motor
    * @param
    *   None
    * @return
    *   Speed, the current speed of the motor
    */
    float GetSpeed(int index);
    
    /** Set the speed (RPM) of the motor
    * @param
    *   SpeedRPM, the Speed in RPM the motor must reach 
    * @return
    *  None
    */
    void SetSpeedA(float SpeedRPM);
    
    /** change the parameters of control
    * @param
    *   GainProp, the value of the propotionnal gain used in the asservissment
    *   GainInt, the value of the integral gain used in the asservissment
    * @return
    *   None
    */
    void SetAsser(float GainProp, float GainInt);
    
    /** Calculate the speed of the motor 
    * @param
    *   None
    * @return
    *   None
    */
    void rising();
    
    /** Return the calcul  ated input
    * @param
    *   None
    *@ @return
    *   percent, the last output calulated by the PI
    */
    float GetU();
    
    /** Return the state
    * @param
    *   None
    *@ @return
    *   state, 1 fwd, 2 rev, 0 stop
    */
    string state();
    
 
  private:
    PwmOut _en;
    DigitalOut _fwd, _rev;
    InterruptIn _chanA;
    DigitalIn _chanB;
    float _currentSpeed;
    float _lastAUp;
    Timer _Time;
    float _gainProp;
    float _gainInt;
    float _maxSpeed;
    float _lastOut;
    
    int _rpm_index;
};
static std::vector<float> rpm;
static int count_ini = 10;

#endif
