/**
 * Copyright (c) 2017, Arm Limited and affiliates.
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
#include <stdio.h>
#include "mbed.h"

// Application helpers

#include "trace_helper.h"

// Application peripherals

#include "serial.h"
#include "gps.h"
#include "lora_radio.h"
#include "SPI_MX25R.h"


using namespace events;


/*
 * Sets up an application dependent transmission timer in ms. Used only when Duty Cycling is off for testing
 */
#define TX_TIMER                        60000

/**
 * Maximum number of events for the event queue.
 * 10 is the safe number for the stack events, however, if application
 * also uses the queue for whatever purposes, this number should be increased.
 */
#define MAX_NUMBER_OF_EVENTS            10

/**
 * Maximum number of retries for CONFIRMED messages before giving up
 */
#define CONFIRMED_MSG_RETRY_COUNTER     3


void serial_post_to_queue(void);

/**
* This event queue is the global event queue for both the
* application and stack. To conserve memory, the stack is designed to run
* in the same thread as the application and the application is responsible for
* providing an event queue to the stack that will be used for ISR deferment as
* well as application information event queuing.
*/
static EventQueue ev_queue(MAX_NUMBER_OF_EVENTS *EVENTS_EVENT_SIZE);

/**
 * Application specific callbacks
 */
static lorawan_app_callbacks_t callbacks;

/**
 * Entry point for application
 */
 
mbed::DigitalOut _alive_led(P1_13, 0);
mbed::DigitalOut _actuated_led(P1_14,1);
int latitude=0;
int longitude=0;


void GPS_Read(void)
{
    uint8_t tx_buffer[256];
    uint16_t packet_len;
    int16_t retcode;
    
    gps_print_local();
 
    packet_len = sprintf((char *) tx_buffer, "%d, %d\n", get_latitude(), get_longitude());
    retcode = lora_send_message(tx_buffer, packet_len);

}

void serial_rx(){
    if(pc.readable()){
        pc.printf("rx: %c\n", pc.getc());
    } 
    pc.attach(&serial_post_to_queue, RawSerial::RxIrq);
    return;
}
 
void serial_post_to_queue(void){
    //disable serial rx interrupt
    pc.attach(NULL, RawSerial::RxIrq);
    //enqueue the serial rx reception as a normal task
    ev_queue.call(SerialRx);
    return;
}

int main(void)
{
    pc.printf("init\n");
    pc.baud(9600);
    pc.printf("config9600\n");
    //enable serial rx interrupt
    
    wait_ms(250);
    
    _actuated_led =0;
    
    uint8_t data[2];
    data[0]= 0xFF;  //byte qualquer
    data[1] = 0xFF; //byte qualquer
    
    SPI_MX25R *flash = new SPI_MX25R(P0_20, P0_17, P0_22, P0_24);
    
    flash->writeEnable();
    flash->programPage(0x191F50, data, 2);
    
    pc.printf("o numero foi: %d\n\r", flash->read8(0x191F50));
    pc.printf("o numero foi: %d\n\r", flash->read8(0x191F51));
    
    return 0;
}


/**
 * Event handler
 */
void lora_event_handler(lorawan_event_t event)
{
    switch (event) {
        case CONNECTED:
            pc.printf("\r\n Connection - Successful \r\n");
            if (MBED_CONF_LORA_DUTY_CYCLE_ON) {
                //send_message();
                //lora_send_message((uint8_t*)"testeLora", (uint16_t)10);
            } //else {
                //ev_queue.call_every(TX_TIMER, (void)lora_send_message((uint8_t*)"testeLoraEvery", (uint16_t)15));
            //}

            break;
        case DISCONNECTED:
            ev_queue.break_dispatch();
            pc.printf("\r\n Disconnected Successfully \r\n");
            break;
        case TX_DONE:
//            printf("\r\n Message Sent to Network Server \r\n");
            if (MBED_CONF_LORA_DUTY_CYCLE_ON) {
                //lora_send_message((uint8_t*)"teste", (uint16_t)6);
            }
            break;
        case TX_TIMEOUT:
//        printf("\r\n Transmission Error TX_Timeout");
            break;
        case TX_ERROR:
//        printf("\r\n Transmission Error TX_Error");
            break;
        case TX_CRYPTO_ERROR:
//        printf("\r\n Transmission Error TX_Crypto_Error");
            break;
        case TX_SCHEDULING_ERROR:
//            printf("\r\n Transmission Error - EventCode = %d \r\n", event);
            // try again
            if (MBED_CONF_LORA_DUTY_CYCLE_ON) {
                //lora_send_message((uint8_t*)"teste2", (uint16_t)7);
            }
            break;
        case RX_DONE:
//            printf("\r\n Received message from Network Server \r\n");
            lora_receive_message();
            break;
        case RX_TIMEOUT:
//        printf("\r\n Transmission Error RX_Timeout");
            break;
        case RX_ERROR:
//            printf("\r\n Error in reception - Code = %d \r\n", event);
            break;
        case JOIN_FAILURE:
//            printf("\r\n OTAA Failed - Check Keys \r\n");
            break;
        case UPLINK_REQUIRED:
//            printf("\r\n Uplink required by NS \r\n");
            if (MBED_CONF_LORA_DUTY_CYCLE_ON) {
                //lora_send_message((uint8_t*)"uplink", (uint16_t)7);
            }
            break;
        default:
            MBED_ASSERT("Unknown Event");
            break;
    }
}

// EOF
