//***************************************************************
//  Function Generator - main entry and sole source file for an
//   MBED based function generator.  The program will output
//   waveforms sinewave, squarewave, sawtooth.
//
//   Author:  Matt Shilling & Terry Richards
//   Date:    9/29/2016
//
#include "mbed.h"
#include "SLCD.h"
#include "TSISensor.h"

SLCD lcd; //create lcd class
TSISensor tsi; // touch sensor
Serial pc(USBTX, USBRX); //init serial

AnalogOut Aout(PTE30); // waveform will be outputted through this pin
DigitalIn sw_right(SW1);
DigitalIn sw_left(SW3);
 
 
bool first = true;
bool just_pressed = false;
int select = 1;

// check_update - checks the buttons to see if the user wants
// to update either the vertical (in volts) or the horizontal
// (in uS) for the currently selected waveform.  It also updates
// the LCD
// **** The joystick is used instead of the pots because *****
// **** I found that the pots were to unstable ***************
// Inputs - the vert and horiz from the calling waveform routine
// Outputs - the updated vert and horiz based on the button presses
//
#define SINE        1
#define SQUARE      2
#define SAW         3


void check_click_right()
{
    if (sw_right == 0)
    {
        just_pressed = true;
    }
    
    if (sw_right == 1 && just_pressed == true)
    {
        select++;
        if (select == 4) select = 1;
        just_pressed = false;
        pc.printf("selection %d \n", select);
    }

}

void check_update(  int type,   //what type of waveform
                    double *vert,   //vertical in volts
                    int *horiz,     //horizontal in microseconds (uS)
                    float vertmax,  //max vertical in volts
                    int horizmax,   //max horizontal in us
                    bool disp_info )     //put initial vert and horiz
{                                   //on LCD screen if set
    if( type ) {    //The header is static once the waveform is chosen
        lcd.clear();
        lcd.Home();
        switch( type ) {    //only done if type is set
            case SINE:
                lcd.printf("sin");
                break;
            case SQUARE:
                lcd.printf("sqr");
                break;
            case SAW:
                lcd.printf("saw");
        }
    }
    
    if(tsi.readPercentage() != 0 
       && tsi.readPercentage() < (1.0 - 0.001)) { 
        
        if (sw_left == 1) {
            *vert = tsi.readPercentage();  //update the vertical
            pc.printf("Height Changed to: %f \n", *vert);
        }
        
    }

    
    if(tsi.readPercentage() != 0
       && sw_left == 0 ) {  //updates the horizontal in us

        *horiz = int(tsi.readPercentage() * 1000);
        pc.printf("Period Changed to: %d \n", *horiz);
    }
    
    if(disp_info) {    //only done if bool is true
        //pc.printf("PW/Period: %d uS \n", *horiz);
        //pc.printf("Height: %5.3f V \n\n", *vert);
    }
    
    check_click_right();
    
}
 
// SineWave - is initiated when the user selects sinewave output
// Sends a sine wave to the Aout pin.
// The Touch Slider controls amplitude
// The Touch Slider + left button controls the period
void sineWave()
{
    int  horiz=1000;
    double vert=1.0, outval, i;
    check_update( SINE, &vert, &horiz, 1.0, 10000, true );
    while(select == SINE) {       // thread loop
        check_update( 0, &vert, &horiz, 1.0, 10000, false );
        for (i=0; i<2; i=i+0.05) {
            outval = 0.5 + 0.5*vert*sin(i*3.14159);
            Aout.write(outval);  // Compute the sine value, + half the range
            wait_us(horiz);                      // Controls the sine wave period
        }
    }
    
    first = true;
}
 
// squareWave - called if user selects squarewave. Sends the
// square wave to the Aout pin.
// The Touch Slider controls amplitude
// The Touch Slider + left button controls the period
void squareWave()
{
    static double height = 1.0;
    static int width = 20;
    check_update( SQUARE, &height, &width, 1.0, 10000, true );
    while(select == SQUARE) {       // thread loop
        check_update( 0, &height, &width, 1.0, 100000, false );
        Aout.write(height);
        wait_us(width);
        Aout.write(0);
        wait_us(width);
    }
    
    first = true;
}

// SawTooth - called if the user selects sawTooth.  Sends the
// saw tooth waveform out to the Aout pin.
// The up and down buttons of the joystick control the amplitude
// The right and left buttons control the period
void sawTooth()
{
    static double height = 1.0, inc;
    static int width = 6000, i;
    inc = height/width;
    check_update( SAW, &height, &width, 1.0, 10000, true );
    while(select == SAW) {       // thread loop
        check_update( 0, &height, &width, 1.0, 100000, false );
        inc = height/width;
        for( i=0; i<width; i++) {       
            Aout.write(i*inc);
            //wait_us(1);
        }
    }
    
    first = true;
}
// Banner - Output a welcome and select screen for
// the user and give them instructions for use
void banner(){
    
    pc.printf("WELCOME TO THE FRDM-KL46Z WAVEFORM GENERATOR \n");
    pc.printf("Writen by Matt Shilling based on code from Terry Richards \n\n");
    pc.printf("---------------------- INSTRUCTIONS ----------------------\n");
    pc.printf("1.) Click right button to select waveform\n");
    pc.printf("2.) Use touch slider to control wave amplitude\n");
    pc.printf("3.) Use touch slider + left putton to control wave period\n");
    
}
 
// main - main entry point to program and where the 
// user selects the desired waveform.   
int main()
{
    //booleans to select waveform
    bool do_sine=false, do_saw=false;
    bool do_square=false; 
    //pc.baud(19200);       //debug
    banner();   //print instructions on serial
    while(1)  {
        if( select == SAW ) {  //is SAW selected?
            do_saw=true;    //select Sawtooth
            do_sine=false;  //ensure nothing else selected
            do_square=false;
        }

        if( select == SQUARE ) {   //is RIGHT pressed?
            do_saw=false;
            do_sine=false;
            do_square=true; //user wants squarewave
        }
        
        if( select == SINE ) {    //is LEFT pressed?
            do_saw=false;
            do_sine=true;   //user wants sinewave
            do_square=false;
        }
 
        if(first ) {  //wave has ended

            if( do_saw ) {
                //pc.printf("I'm doing saw\r\n");
                sawTooth();
            }
            else if( do_square ) {
                //pc.printf("I'm doing square\r\n");
                squareWave();
            }
            else if( do_sine ) {
                //pc.printf("I'm doing sine\r\n");
                sineWave();
            }
            else {  //Default if no button pushed
                //pc.printf("I'm doing default (sine)\r\n");
                sineWave();
            }
            //banner();   //we came back, ask user what next
        }
    }
}