/* Copyright (c) 2010-2011 mbed.org, MIT License
*
* Permission is hereby granted, free of charge, to any person obtaining a copy of this software
* and associated documentation files (the "Software"), to deal in the Software without
* restriction, including without limitation the rights to use, copy, modify, merge, publish,
* distribute, sublicense, and/or sell copies of the Software, and to permit persons to whom the
* Software is furnished to do so, subject to the following conditions:
*
* The above copyright notice and this permission notice shall be included in all copies or
* substantial portions of the Software.
*
* THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, INCLUDING
* BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
* NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM,
* DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
* OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
*/

#ifndef FXAS21000_H
#define FXAS21000_H

#include "mbed.h"
#include "MotionSensor.h"

/**
 * FXAS21000 Slave Address
 */
#define FXAS21000_SLAVE_ADDR 0x40

// FXAS21000 internal register addresses
#define FXAS21000_STATUS 0x00
#define FXAS21000_WHOAMI 0x0C
#define FXAS21000_XYZ_DATA_CFG 0x0E
#define FXAS21000_CTRL_REG0 0x0D
#define FXAS21000_CTRL_REG1 0x13
#define FXAS21000_WHOAMI_VAL 0xD1
#define FXAS21000_OUT_X_MSB 0x01
#define FXAS21000_OUT_Y_MSB 0x03
#define FXAS21000_OUT_Z_MSB 0x05

/**
 * FXAS21000 driver class
 */
class FXAS21000 : public MotionSensor
{
public:

    /** Read a device register
        @param addr The address to read from
        @param data The data to read from it
        @param len The amount of data to read from it
        @return 0 if successful, negative number otherwise
     */
    void readRegs(uint8_t addr, uint8_t *data, uint32_t len) const;

    /** Read the ID from a whoAmI register
        @return The device whoAmI register contents
     */
    uint8_t whoAmI(void) const;

    virtual void enable(void) const;
    virtual void disable(void) const;
    virtual uint32_t sampleRate(uint32_t frequency) const;
    virtual uint32_t dataReady(void) const;

protected:
    I2C *_i2c;
    uint8_t _addr;
    
    /** FXAS21000 constructor
        @param i2c a configured i2c object
        @param addr addr of the I2C peripheral as wired
     */
    FXAS21000(I2C &i2c, uint8_t addr);

    /** FXAS21000 deconstructor
     */
    ~FXAS21000();
    
    void writeRegs(uint8_t *data, uint32_t len) const;
    int16_t getSensorAxis(uint8_t addr) const;
};

/** FXAS21000Gyroscpe interface
 */
class FXAS21000Gyroscpe : public FXAS21000
{
public:

    FXAS21000Gyroscpe(I2C &i2c, uint8_t addr) : FXAS21000(i2c, addr) {}

    virtual int16_t getX(int16_t &x) const;
    virtual int16_t getY(int16_t &y) const;
    virtual int16_t getZ(int16_t &z) const;
    virtual float getX(float &x) const;
    virtual float getY(float &y) const;
    virtual float getZ(float &z) const;
    virtual void getAxis(motion_data_counts_t &xyz) const;
    virtual void getAxis(motion_data_units_t &xyz) const;

};

#endif
