/**
@file main.h
@brief Header file containing function prototypes, defines and global variables for the main.cpp file.
@brief Version 1.0.0
@author Marty Laverick
@date 5th May 2016
*/

#ifndef _MAIN_H
#define _MAIN_H

#include "mbed.h"
#include "N5110.h"
#include "SDFileSystem.h"
#include "drawings.h"
#include "music.h"

#define DIRECTION_TOLERANCE 0.05f
#define NX 84
#define NY 48

/**
@namespace r_led
@brief GPIO output for the on-board red LED
*/
DigitalOut r_led(LED_RED);
/**
@namespace g_led
@brief GPIO output for the on-board green LED
*/
DigitalOut g_led(LED_GREEN);
/**
@namespace b_led
@brief GPIO output for the on-board blue LED
*/
DigitalOut b_led(LED_BLUE);
/**
@namespace sw2
@brief GPIO input for the on-board switch
*/
InterruptIn sw2(SW2);
/**
@namespace sw3
@brief GPIO input for the on-board switch
*/
InterruptIn sw3(SW3);
/**
@namespace js_pot_x
@brief GPIO input for the x-axis potentiometer on the joystick
*/
AnalogIn js_pot_x(PTB2);
/**
@namespace js_pot_y
@brief GPIO input for the y-axis potentiometer on the joystick
*/
AnalogIn js_pot_y(PTB3);
/**
@namespace js_button
@brief GPIO input for the button on the joystick
*/
InterruptIn js_button(PTB11);
/**
@namespace b_button
@brief GPIO input for the button on the PCB
*/
InterruptIn b_button(PTB10);
/**
@namespace lcd
@brief GPIO inputs and outputs for the visual display
*/
N5110 lcd(PTE26, PTA0, PTC4, PTD0, PTD2, PTD1, PTC3);
/**
@namespace sd
@brief GPIO inputs and outputs for the SD card
*/
SDFileSystem sd(PTE3, PTE1, PTE2, PTE4, "sd");
/**
@namespace update_flag
@brief Ticker to execute the flag update
*/
Ticker update_flag;
/**
@namespace *fp
@brief FILE pointer to allow file access from the SD card
*/
FILE *fp;

float refresh_rate; /*!< refresh_rate set in main */
float g_dt; /*!< g_dt set in main from refresh_rate */
bool up_flag;   /*!< up_flag set in ISR */
bool js_button_previous;    /*!< js_button_previous set and flipped in update_joystick */
bool b_button_previous; /*!< b_button_previous set and flipped in update_joystick */
bool menu_viewed;   /*!< menu_viewed set and flipped in menu */
bool jumping;   /*!< jumping set in init_game and flipped in start_game */
bool spike1_there;  /*!< spike1_there set in init_game and flipped in start_game */
bool spike2_there;  /*!< spike2_there set in init_game and flipped in start_game */
bool spike3_there;  /*!< spike3_there set in init_game and flipped in start_game */
bool spike4_there;  /*!< spike4_there set in init_game and flipped in start_game */
bool spike5_there;  /*!< spike5_there set in init_game and flipped in start_game */
bool dead;  /*!< dead set in init_game and flipped in start_game */
int selector;   /*!< selector set in menu */
int option; /*!< option set in menu */
int health; /*!< health set in init_game and start_game */
int spike_x1;   /*!< spike_x1 set in init_game and start_game */
int spike_x2;   /*!< spike_x2 set in init_game and start_game */
int spike_x3;   /*!< spike_x3 set in init_game and start_game */
int spike_x4;   /*!< spike_x4 set in init_game and start_game */
int spike_x5;   /*!< spike_x5 set in init_game and start_game */
int spike1_counter; /*!< spike1_counter set in init_game and start_game */
int spike2_counter; /*!< spike2_counter set in init_game and start_game */
int spike3_counter; /*!< hspike3_counterealth set in init_game and start_game */
int spike4_counter; /*!< spike4_counter set in init_game and start_game */
int spike5_counter; /*!< spike5_counter set in init_game and start_game */
int jump_counter;   /*!< jump_counter set in init_game and start_game */
int score_counter;  /*!< score_counter set in init_game and start_game */
int highscore1; /*!< highscore1 set in menu and new_highscore */
int highscore2; /*!< highscore2 set in menu and new_highscore */
int highscore3; /*!< highscore3 set in menu and new_highscore */
int highscore4; /*!< highscore4 set in menu and new_highscore */
int highscore5; /*!< highscore5 set in menu and new_highscore */
int letter; /*!< letter set in register_name */
int letter1;    /*!< letter1 set in register_name */
int letter2;    /*!< letter2 set in register_name */
int letter3;    /*!< letter3 set in register_name */
int letter_counter; /*!< letter_counter set in register_name */
char score[10]; /*!< score set in death */
char score1[10];    /*!< score1 set in highscores */
char score2[10];    /*!< score2 set in highscores */
char score3[10];    /*!< score3 set in highscores */
char score4[10];    /*!< score4 set in highscores */
char score5[10];    /*!< score5 set in highscores */
char name[3];   /*!< name set in register_name */
char name1[4];  /*!< name1 set in menu and new_highscore */
char name2[4];  /*!< name2 set in menu and new_highscore */
char name3[4];  /*!< name3 set in menu and new_highscore */
char name4[4];  /*!< name4 set in menu and new_highscore */
char name5[4];  /*!< name5 set in menu and new_highscore */

// enums
enum DirectionName {
    UP,
    DOWN,
    LEFT,
    RIGHT,
    CENTRE,
    UPRIGHT,
    UPLEFT,
    DOWNRIGHT,
    DOWNLEFT,
    UNKNOWN
};

// structs
struct vector_t {
    vector_t() {};
    float x;
    float y;
};
vector_t pos;  // player position
vector_t vel;  // player velocity
vector_t acc;  // player acceleration

struct joystick {
    joystick() {};
    float x;    // current x value
    float x0;   // 'centred' x value
    float y;    // current y value
    float y0;   // 'centred' y value
    int button; // button selector (assume pull-down used, so 1 = pressed, 0 = unpressed)
    int bbutton;
    DirectionName direction;  // current direction
};
joystick joystick;

/**
Calibrates the joystick from it's initial position
*/
void calibrate_joystick();
/**
Sets to up_flag to true
*/
void update_flag_func();
/**
Sets the new values for the joystick struct variables
*/
void update_joystick();
/**
Checks if the js_button has been held down
@returns false if the button has already been registered as presed but is being held down else true
*/
bool js_button_check();
/**
Checks if the b_button has been held down
@returns false if the button has already been registered as presed but is being held down else true
*/
bool b_button_check();
/**
Initialises the settings of on-board components for the K64F
*/
void init_K64F();
/**
Initialises the settings of the N5110
*/
void init_N5110();
/**
Initialises the objects with bitmaps
*/
void init_obj();
/**
Runs the menu screen and allows for progression to further game functions
*/
void menu();
/**
Runs the function as declared by the variable option set in th menu
*/
void menu_select();
/**
Runs the game and calls all other functions relevant to the game
*/
void start_game();
/**
Initialises all variables used in the game and sets the initial screen
*/
void init_game();
/**
Initialises the players position and speed
*/
void init_player();
/**
Sets the health animations in the HEALTH sub-array
*/
void draw_health();
/**
Determines whether the player has collided with the spikes
*/
void check_collisions();
/**
Updates the players position depending on the players current position, velocity and acceleration, all determined from SUVAT equations
*/
void update_physics_engine();
/**
Updates the lcd to show the new game state
*/
void redraw();
/**
Ends the game and shows the user their score when they run out of lives
*/
void death();
/**
Displays the five highest scores and the name of the person who got the score
*/
void highscores();
/**
Determines whether the latest score after death is higher than any current highscores and sets it accordingly
*/
void new_highscore();
/**
Allows the user to resgister a name to accompany their new highscore
*/
void register_name();
/**
Displays an overview of the games concept
*/
void rules();
/**
Displays the details of the games creator
*/
void credits();
/**
@namespace buzzer
@brief GPIO output for music
*/
speaker buzzer(PTA2);
/**
@namespace screen
@brief A three dimensional array for allowing seperate sections of the two dimensional lcd
*/
Screen screen;
/**
@namespace pointer_obj
@brief Object for the pointer array
*/
object pointer_obj(5,5);
/**
@namespace player_still_obj
@brief Object for the player_still array
*/
object player_still_obj(13,15);
/**
@namespace player_moving1_obj
@brief Object for the player_moving1 array
*/
object player_moving1_obj(13,15);
/**
@namespace player_moving2_obj
@brief Object for the player_moving2 array
*/
object player_moving2_obj(13,15);
/**
@namespace player_jumping_obj
@brief Object for the player_jumping array
*/
object player_jumping_obj(13,15);
/**
@namespace spike_small_obj1
@brief Object for the spike_small array
*/
object spike_small_obj1(5,5);
/**
@namespace spike_small_obj2
@brief Object for the spike_small array
*/
object spike_small_obj2(5,5);
/**
@namespace spike_small_obj3
@brief Object for the spike_small array
*/
object spike_small_obj3(5,5);
/**
@namespace spike_large_obj1
@brief Object for the spike_large array
*/
object spike_large_obj1(6,8);
/**
@namespace spike_large_obj2
@brief Object for the spike_large array
*/
object spike_large_obj2(6,8);
/**
@namespace player
@brief Animation for the player_still array, player_moving1 array and player_moving2 array
*/
animation player(13,15,player_still,player_moving1,player_moving2);
/**
@namespace bat1
@brief Animation for the bat_middle array, bat_open array and bat_close array
*/
animation bat1(13,7,bat_middle,bat_open,bat_close);
/**
@namespace bat2
@brief Animation for the bat_middle array, bat_open array and bat_close array
*/
animation bat2(13,7,bat_middle,bat_open,bat_close);
/**
@namespace bat3
@brief Animation for the bat_middle array, bat_open array and bat_close array
*/
animation bat3(13,7,bat_middle,bat_open,bat_close);
/**
@namespace bat4
@brief Animation for the bat_middle array, bat_open array and bat_close array
*/
animation bat4(13,7,bat_middle,bat_open,bat_close);
/**
@namespace bat5
@brief Animation for the bat_middle array, bat_open array and bat_close array
*/
animation bat5(13,7,bat_middle,bat_open,bat_close);
/**
@namespace background
@brief Background to add scenery to the BG sub-array
*/
Background background;

#endif