/***************************************************************************
 * STM32 VGA demo
 * Copyright (C) 2012 Artekit Italy
 * http://www.artekit.eu
 * Originally Written by Ruben H. Meleca
 * updated for STM32F3 discovery with colour VGA output
 
### video.c
 
#   This program is free software; you can redistribute it and/or modify
#   it under the terms of the GNU General Public License as published by
#   the Free Software Foundation; either version 2 of the License, or
#   (at your option) any later version.
#
#   This program is distributed in the hope that it will be useful,
#   but WITHOUT ANY WARRANTY; without even the implied warranty of
#   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#   GNU General Public License for more details.
#
#   You should have received a copy of the GNU General Public License
#   along with this program; if not, write to the Free Software
#   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

***************************************************************************/

// VGA connector
// ---------------------
// \  Ro Go Bo  o Xo   /	X=Gnd
//  \ o  o  o  o Xo   /		H=HSync=PA8	R=Red=PD0
//   \ o  o Ho Vo  o /		V=Vsync=PA1 G=Green=PD1
//    --------------		            B=Blue=PD2
// 8x2 connector
// R G B o   o X X 
// o o o o o H V o
// 
#include "stm32f30x.h"
#include "video.h"

__attribute__ ((aligned (4))) u8 volatile fba[VID_VSIZE*HTOTAL];	/* Frame buffer */

__attribute__ ((section ("ccmram_data"))) u8 volatile *fb[VID_VSIZE];

int fboffset=0;

__attribute__ ((section ("ccmram_data"))) static volatile u16 vline = 0;				/* The current line being drawn */
__attribute__ ((section ("ccmram_data"))) static volatile u32 vflag = 0;				/* When 1, the DMA request can draw on the screen */
__attribute__ ((section ("ccmram_data"))) static volatile u32 vdraw = 0;				/* Used to increment vline every 3 drawn lines */

#define GPIO_MODE_INPUT 0
#define GPIO_MODE_OUTPUT 1
#define GPIO_MODE_AF 2

#define GPIO_NO_PULL 0
#define GPIO_PULL_UP 1
#define GPIO_PULL_DOWN 2

#define GPIO_OUTPUT_PUSH_PULL 0


#define GPIO_SPEED_MEDIUM 1
#define GPIO_SPEED_LOW 0
#define GPIO_SPEED_HIGH 3


void gpio_set_mode(GPIO_TypeDef *g,int n,int mode) {
	n=n<<1;
	g->MODER = (g->MODER & ~(3<<n)) | (mode<<n);
}

void gpio_set_af(GPIO_TypeDef *g,int n,int af, int otype, int pupd, int speed) {
	int reg=n>>3;
	int pos=(n&7)*4;
	g->AFR[reg] = (g->AFR[reg] & ~(0xf<<pos)) | (af<<pos); // alt func
	pos=(n<<1);
	g->OSPEEDR = (g->OSPEEDR & ~(3<<pos)) | (speed<<pos);
	g->OTYPER = (g->OTYPER & ~(1<<n)) | (otype<<n);
	gpio_set_mode(g,n,GPIO_MODE_AF);
	g->PUPDR = (g->PUPDR & ~(3<<pos)) | (pupd<<pos);	
}

void TIMER_Configuration(void) {
	u32 TimerPeriod = 0;
	u16 Channel1Pulse = 0, Channel2Pulse = 0, Channel3Pulse = 0;
	
	gpio_set_af(GPIOA,1,1,GPIO_OUTPUT_PUSH_PULL, GPIO_NO_PULL, GPIO_SPEED_LOW);
	gpio_set_af(GPIOA,8,6,GPIO_OUTPUT_PUSH_PULL, GPIO_NO_PULL, GPIO_SPEED_LOW);

	/*
		VGA 640x480 @ 60 Hz
		Vertical refresh	31.46875 kHz
		Pixel freq.			36.0 MHz
		
		1 system tick @ 72Mhz = 0,0138 us
	*/
	
	/*
		Horizontal timing
		-----------------
		
		Timer 1 period = 31468.53 Hz		
	
		Horizontal timing info
		----------------------

						Dots	us					ticks
		----------------------------------------------------		
		Visible area	640		25.422045680238		1830
		Front porch		16		0.6355511420059		46
		Sync pulse		96		3.8133068520357		275
		Back porch		48		1.9066534260179		137
		Whole line		800		31.777557100298		2288
		Sync + bp		144		5.7199602780536		412
	
	*/

	TimerPeriod = 2288;//2303;
	Channel1Pulse = 275;//274;//274;//277;		/* HSYNC */
	Channel3Pulse = 412;//412;//394;
	
	TIM1->CR1 &= ~TIM_CR1_CEN;
	
	TIM1->PSC=0;
	TIM1->ARR=TimerPeriod;
	TIM1->CNT=0;
	
	// disable Capture and Compare
	TIM1->CCER &= ~(TIM_CCER_CC1E);
	// set output compare 1 to PWM mode with preload
	TIM1->CCMR1 = (TIM1->CCMR1 & ~(TIM_CCMR1_OC1M | TIM_CCMR1_CC1S)) | TIM_CCMR1_OC1M_2 | TIM_CCMR1_OC1M_1;
	TIM1->CCR1=Channel1Pulse;
	// enable Capture and Compare 1
	TIM1->CCER |=  TIM_CCER_CC1E | TIM_CCER_CC1P ; // output polarity low
	// main output enable
    TIM1->BDTR |= TIM_BDTR_MOE; 
	
	TIM1->CR2 = (TIM1->CR2 & ~TIM_CR2_MMS) | (2<<4); // TIM_TRGOSource_Update mode
	
	TIM8->CR1 &= ~TIM_CR1_CEN;
	TIM8->CR1 |= TIM_CR1_ARPE | TIM_CR1_URS;
	TIM8->PSC=0;
	TIM8->ARR=8;
	TIM8->CNT=0;
	
	TIM8->DIER |= TIM_DIER_UDE;
	
	TIM8->SMCR=TIM8->SMCR & ~(TIM_SMCR_SMS | TIM_SMCR_TS) | 6 | (3<<4); // trigger mode from itr3 (tim3)
	
	TIM3->CR1 &= ~TIM_CR1_CEN;
	TIM3->PSC=0;
	TIM3->ARR=Channel3Pulse;
	TIM3->CNT=0;
	TIM3->CR1 |= TIM_CR1_ARPE | TIM_CR1_URS;
	TIM3->CR1 |= TIM_CR1_OPM; 
	TIM3->SMCR=TIM3->SMCR & ~(TIM_SMCR_SMS | TIM_SMCR_TS) | 6 ; // trigger mode from itr0 (tim1)
	TIM3->CR2 = (TIM3->CR2 & ~TIM_CR2_MMS) | (2<<4); ///*TIM_CR2_MMS_2 |*/ TIM_CR2_MMS_1 /*| TIM_CR2_MMS_0*/;// TIM_TRGOSource_Update mode

	/*
		Vertical timing
		---------------
		
		Polarity of vertical sync pulse is negative.

						Lines
		------------------------------
		Visible area	480
		Front porch		10
		Sync pulse		2
		Back porch		33
		Whole frame		525
		
	*/

	/* VSYNC (TIM2_CH2) and VSYNC_BACKPORCH (TIM2_CH3) */
	/* Channel 2 and 3 Configuration in PWM mode */
	TIM2->SMCR=TIM2->SMCR & ~(TIM_SMCR_SMS | TIM_SMCR_TS) | 5 ;// gated slave mode trigger source 0
	
	
	TimerPeriod = 525;		/* Vertical lines */
	Channel2Pulse = 2;		/* Sync pulse */
	Channel3Pulse = 35;		/* Sync pulse + Back porch */

	TIM2->CR1 &= ~TIM_CR1_CEN;
	TIM2->PSC=0;
	TIM2->ARR=TimerPeriod;
	TIM2->CNT=0;

	// disable Capture and Compare 2 and 3
	TIM2->CCER &= ~(TIM_CCER_CC2E | TIM_CCER_CC3E);
	// set output compare 1 to PWM mode with preload
	TIM2->CCMR1 = (TIM2->CCMR1 & ~(TIM_CCMR1_OC2M | TIM_CCMR1_CC2S)) | TIM_CCMR1_OC2M_2 | TIM_CCMR1_OC2M_1;
	TIM2->CCR2=Channel2Pulse;
	TIM2->CCR3=Channel3Pulse;
	
	// enable Capture and Compare 2 and 3
	TIM2->CCER |=  TIM_CCER_CC2E |  TIM_CCER_CC3E | TIM_CCER_CC1P | TIM_CCER_CC2P; // output polarity low
	
	// main output enable
    TIM2->BDTR |= TIM_BDTR_MOE; 

	NVIC->IP[TIM2_IRQn]=32; // Interrupt Priority, lower is higher priority
	NVIC->ISER[TIM2_IRQn >> 0x05] = 1 << (TIM2_IRQn & 0x1F); // Interrupt enable
	
	TIM2->DIER |= TIM_DIER_CC3IE;

	TIM2->CR1 |= TIM_CR1_CEN;
	TIM1->CR1 |= TIM_CR1_CEN;
}

void DMA_Configuration(void) {	
	RCC->APB1ENR |= RCC_APB1ENR_PWREN | RCC_APB1ENR_TIM2EN | RCC_APB1ENR_TIM3EN;
	RCC->APB2ENR |= RCC_APB2ENR_SYSCFGEN | RCC_APB2ENR_TIM1EN | RCC_APB2ENR_TIM8EN ;
	RCC->AHBENR |= RCC_AHBENR_GPIOBEN | RCC_AHBENR_GPIOAEN | RCC_AHBENR_GPIOEEN | RCC_AHBENR_GPIODEN;

	GPIOD->MODER = (GPIOD->MODER&0xffff0000) | 0x5555; // output mode for PD0-7
	GPIOD->PUPDR = (GPIOD->PUPDR & ~0xffff);// | 0xaaaa; // pull down (1010)
//	GPIOD->OTYPER |= 0x8;
	GPIOD->OSPEEDR = (GPIOD->OSPEEDR & ~0xffff) | 0xffff;
	
	RCC->AHBENR |= RCC_AHBENR_DMA2EN;
    // direction = peripheral dest, memory inc, peripheral size=halfword, memory size=byte, priority level=high, transmission complete interrupt enabled
    DMA2_Channel1->CCR = DMA_CCR_DIR | DMA_CCR_MINC  | DMA_CCR_PL_1 | DMA_CCR_PL_0 | DMA_CCR_TCIE;
    // bytes to transfer
    DMA2_Channel1->CNDTR = HTOTAL;
    // peripheral address
    DMA2_Channel1->CPAR =(uint32_t) &GPIOD->ODR;
    // memory address
    DMA2_Channel1->CMAR =(u32) fba+fboffset;
	// configure NVIC
    NVIC->IP[DMA2_Channel1_IRQn]=16; // Interrupt Priority, lower is higher priority
	NVIC->ISER[DMA2_Channel1_IRQn >> 0x05] = 1 << (DMA2_Channel1_IRQn & 0x1F); // Interrupt enable
	
}

//*****************************************************************************
//	This irq is generated at the end of the vertical back porch.
//	Sets the 'vflag' variable to 1 (valid vertical frame).
//*****************************************************************************

uint32_t dummy=0;
__attribute__ ((section ("ccmram"))) void TIM2_IRQHandler(void) {
	vflag = 1;
	TIM2->SR = 0xFFF7; //~TIM_IT_CC3;
	TIM8->CR1 &= ~TIM_CR1_CEN;
	DMA2_Channel1->CCR = DMA_CCR_DIR | DMA_CCR_MINC  | DMA_CCR_PL_1 | DMA_CCR_PL_0 | DMA_CCR_TCIE ;
	DMA2_Channel1->CNDTR = HTOTAL;
	TIM8->CNT=0;
	DMA2_Channel1->CPAR=(uint32_t)&dummy;//(uint32_t) &GPIOD->ODR+1;
	GPIOD->ODR=0x00;
	DMA2_Channel1->CCR = DMA_CCR_DIR  | DMA_CCR_MINC  | DMA_CCR_PL_1 | DMA_CCR_PL_0 | DMA_CCR_TCIE | DMA_CCR_EN;// | DMA_CCR_CIRC;
}

//*****************************************************************************
//	This interrupt is generated at the end of every line.
//	It will increment the line number and set the corresponding line pointer
//	in the DMA register.
//*****************************************************************************

__attribute__ ((section ("ccmram"))) void DMA2_Channel1_IRQHandler(void) {	
	DMA2->IFCR = DMA_ISR_TCIF1;
	TIM8->CR1 &= ~TIM_CR1_CEN;
	TIM8->CNT=0;
	DMA2_Channel1->CCR = DMA_CCR_DIR | DMA_CCR_MINC  | DMA_CCR_PL_1 | DMA_CCR_PL_0 | DMA_CCR_TCIE;
	DMA2_Channel1->CNDTR = HTOTAL;
	DMA2_Channel1->CPAR=(uint32_t) &GPIOD->ODR;;
	vdraw++;
	if (vdraw == 3) {
		vdraw = 0;
		vline++;
		if (vline == VID_VSIZE) {
			vdraw = vline = vflag = 0;
			DMA2_Channel1->CMAR = (u32)fba;
			DMA2_Channel1->CPAR=(uint32_t)&dummy;
			GPIOD->ODR=0x00;
		} else {
			DMA2_Channel1->CMAR += HTOTAL;
		}
	}
	if(vflag)
		DMA2_Channel1->CCR = DMA_CCR_DIR | DMA_CCR_MINC  | DMA_CCR_PL_1 | DMA_CCR_PL_0 | DMA_CCR_TCIE | DMA_CCR_EN;
	
	
}

__attribute__ ((section ("ccmram"))) void vidNextBuffer(void) {
	    unsigned *fp=(unsigned *)fba;
	    for(int i=0;i<VID_VSIZE*HTOTAL/4;i++)
	    	*fp++=(*fp>>4)&0xf0f0f0f;
}

__attribute__ ((section ("ccmram"))) void waitForRefresh(void) {
	while(vflag) __WFI();
}

void vidClearScreen(void) {
	u16 x, y;

	for (y = 0; y < VID_VSIZE; y++) 	{
		for (x = 0; x < HTOTAL; x++) {
			fba[y*HTOTAL+x] = 0;
		}
	}
}

void vidDot(int x, int y, int col) {
	fb[y][x]=col;
}

void vidLine(int x, int y, int col) {
	fb[y][x]=col;
}

void vidInit(void) {
	int i;
	DMA_Configuration();
	TIMER_Configuration();
	for(i=0;i<VID_VSIZE;i++)
		fb[i]=fba+i*HTOTAL+2;
	vidClearScreen();
}

