/**
 * =============================================================================
 * Firmware updater (Version 0.0.1)
 * =============================================================================
 * Copyright (c) 2010 Shinichiro Nakamura (CuBeatSystems)
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 * =============================================================================
 */

#ifndef FIRMWARE_UPDATER_H
#define FIRMWARE_UPDATER_H

#include "mbed.h"
#include "LocalFileSystem.h"
#include "MSCFileSystem.h"
#include <string>

/**
 * @code
 * #include "mbed.h"
 * #include "FirmwareUpdater.h"
 * #include "EthernetNetIf.h"
 *
 * EthernetNetIf eth;
 * FirmwareUpdater fwup("http://mbed.org/media/uploads/shintamainjp/", "firmware", true);
 *
 * // There are 2 files for the firmware.
 * //  1. firmware.txt : firmware version file.
 * //  2. firmware.bin : firmware binary file.
 *
 * int main() {
 *     eth.setup();
 *     if (fwup.exist() == 0) {
 *         printf("Found a new firmware.\n");
 *         if (fwup.execute() == 0) {
 *             printf("Update succeed.\n");
 *             printf("Resetting this system...\n\n\n\n\n");
 *             fwup.reset();
 *         } else {
 *             printf("Update failed!\n");
 *         }
 *     }
 * }
 * @endcode
 */
class FirmwareUpdater {
public:
    /**
     * Create.
     *
     * @param url URL for firmware. Do not include a target file name.
     * @param name An application name. Do not include a extention.
     * @param log True if logging.
     */
    FirmwareUpdater(std::string src_name, std::string dest_name, bool log = false);

    /**
     * Dispose.
     */
    ~FirmwareUpdater();

    /**
     * Get a URL.
     *
     * @return URL.
     */
    const std::string get_src_name() const;

    /**
     * Get a name.
     *
     * @return name.
     */
    const std::string get_dest_name() const;

    /**
     * Checking a new firmware.
     * Compare versions of the software between local storage on mbed and on webserver.
     *
     * @return Return 0 if a new firmware exists.
     */
    int exist();

    /**
     * Execute update.
     *
     * @return Return 0 if it succeed.
     */
    int execute();

    /**
     * Reset system.
     */
    void reset();

private:
    static const std::string EXT_BIN;
    static const std::string EXT_BINTMP;
    static const std::string EXT_TXT;
    static const std::string EXT_TXTTMP;
    const std::string src_name;
    const std::string dest_name;
    const bool log;
    LocalFileSystem local;
   
    
    /**
     * Fetch a file.
     *
     * @param src_url URL of a source file.
     * @param local_file Local file name.
     *
     * @return Return 0 if it succeed.
     */
    int fetch(std::string src_url, std::string local_file);

    /**
    * Clean up all bin files.
    */
    int cleanupAllBinFiles(void);

    /**
     * Copy a file.
     *
     * @param local_file1 Source file.
     * @param local_file2 Destination file.
     *
     * @return Return 0 if it succeed.
     */
    int copy(std::string local_file1, std::string local_file2);

    /**
     * Output a message to a log file.
     *
     * @param format ...
     */
    void LOG(const char* format, ...);

};

#endif
