/**
  * =============================================================================
  * Firmware updater (Version 0.0.1)
  * =============================================================================
  * Copyright (c) 2010 Shinichiro Nakamura (CuBeatSystems)
  *
  * Permission is hereby granted, free of charge, to any person obtaining a copy
  * of this software and associated documentation files (the "Software"), to deal
  * in the Software without restriction, including without limitation the rights
  * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
  * copies of the Software, and to permit persons to whom the Software is
  * furnished to do so, subject to the following conditions:
  *
  * The above copyright notice and this permission notice shall be included in
  * all copies or substantial portions of the Software.
  *
  * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
  * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
  * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
  * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
  * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
  * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
  * THE SOFTWARE.
  * =============================================================================
  */

#include "FirmwareUpdater.h"

#include <stdio.h>
#include <stdarg.h>

extern "C" void mbed_reset();

const std::string FirmwareUpdater::EXT_BIN = ".bin";
const std::string FirmwareUpdater::EXT_BINTMP = ".b__";
const std::string FirmwareUpdater::EXT_TXT = ".txt";
const std::string FirmwareUpdater::EXT_TXTTMP = ".t__";

MSCFileSystem fs ("fs");
LocalFileSystem local("local");


/**
 * Create.
 *
 * @param src_name: An application name on USB Stick. Do not include a extention.
 * @param dest_name An application name on MBED. Do not include a extention.
 * @param log True if logging.
 */
FirmwareUpdater::FirmwareUpdater(std::string src_name, std::string dest_name, bool log)
        : src_name(src_name), dest_name(dest_name), log(log), local("local") {

}

/*
 * Dispose.
 */
FirmwareUpdater::~FirmwareUpdater() {
}

/**
 * Get the source_name.
 *
 * @return src_name.
 */
const std::string FirmwareUpdater:: get_src_name() const {
    return src_name;
}

/**
 * Get the dest_name.
 *
 * @return _destname.
 */
const std::string FirmwareUpdater:: get_dest_name() const {
    return dest_name;
}

/*
 * Checking a new firmware.
 * Compare versions of the software between local storage on mbed and on USB Stick.
 *
 * @return Return 0 if a new firmware exists.
 */

int FirmwareUpdater::exist() {
    int ver_local, ver_USB;
    printf("Fetch the version from a local.");
    /*
     * Fetch the version from a local.
     */
    std::string file_local = "/local/" + dest_name + EXT_TXT;
    FILE *fp = fopen(file_local.c_str(), "rb");
    if (fp == NULL) {
        LOG("Local firmware version file '%s' open failed.\n\r", file_local.c_str());
        return -1;
    }
    if (fscanf(fp, "%d", &ver_local) != 1) {
        LOG("Local firmware version file '%s' is invalid.\n\r", file_local.c_str());
        return -2;
    }
    fclose(fp);
    LOG("Local firmware version is %d. (%s)\n\r", ver_local, file_local.c_str());

    /*
     * Fetch the version from  USB Stick.
     */
     printf("Fetch the version from  USB Stick.\r");
    std::string file_USB = "/fs/" + src_name + EXT_TXT;
    fp = fopen(file_USB.c_str(), "rb");
    
    if (fp == NULL) {
        LOG("Source firmware version file '%s' open failed.\n\r", file_USB.c_str());
        return -1;
    }
    if (fscanf(fp, "%d", &ver_USB) != 1) {
        LOG("Source firmware version file '%s' is invalid.\n\r", file_USB.c_str());
        return -2;
    }
    fclose(fp);
    LOG("Source firmware version is %d. (%s)\n\r", ver_USB, file_USB.c_str());



    return (ver_local < ver_USB) ? 0 : 1;
}

/**
 * Execute update.
 *
 * @return Return 0 if it succeed.
 */
int FirmwareUpdater::execute() {
    /*
     * Fetch the files.
     */
    printf("Start execute\r");
    std::string USB_txt = "/fs/" + src_name + EXT_TXT;
    std::string USB_bin = "/fs/" + src_name + EXT_BIN;
    
   /*
    std::string file_txttmp = "/local/" + name + EXT_TXTTMP;
    std::string file_bintmp = "/local/" + name + EXT_BINTMP;
    if (fetch(serv_txt, file_txttmp) != 0) {
        return -1;
    }
    if (fetch(serv_bin, file_bintmp) != 0) {
        return -2;
    }*/
    cleanupAllBinFiles();
    /*
     * Copy it.
     */
     
    std::string local_txt = "/local/" + dest_name + EXT_TXT;
    std::string local_bin = "/local/" + dest_name + EXT_BIN;
    if (copy(USB_txt, local_txt) != 0) {
        printf("-3");
        return -3;
    }
    if (copy(USB_bin, local_bin) != 0) {
        printf("-4");
        return -4;
    }
    /*
     * Delete the temporary files.
     */
    //remove(file_txttmp.c_str());
    //remove(file_bintmp.c_str());
    return 0;
}


/**
 * Reset system.
 */
void FirmwareUpdater::reset() {
    mbed_reset();
}

/**
 * Fetch a file.
 *
 * @param src_url URL of a source file.
 * @param local_file Local file name.
 *
 * @return Return 0 if it succeed.
 */
int FirmwareUpdater::fetch(std::string src_name, std::string local_file) {
    return (0);
}


/**
  * Copy a file.
  *
  * @param local_file1 Source file.
  * @param local_file2 Destination file.
  *
  * @return Return 0 if it succeed.
  */
int FirmwareUpdater::copy(std::string local_file1, std::string local_file2) {
    
    LOG("File copying... (%s->%s)\n\r", local_file1.c_str(), local_file2.c_str());

    FILE *rp = fopen(local_file1.c_str(), "rb");
    if (rp == NULL) {
        LOG("File '%s' open failed.\n", local_file1.c_str());
        return -1;
    }
    remove(local_file2.c_str());
    FILE *wp = fopen(local_file2.c_str(), "wb");
    if (wp == NULL) {
        LOG("File '%s' open failed.\n\r", local_file2.c_str());
        fclose(rp);
        return -2;
    }
    int c;
    while ((c = fgetc(rp)) != EOF) {
        fputc(c, wp);
    }
    fclose(rp);
    fclose(wp);
    LOG("File copied. (%s->%s)\n\r", local_file1.c_str(), local_file2.c_str());
    return 0;
}
/**
 * Clean up all bin files.
 */
int FirmwareUpdater::cleanupAllBinFiles(void) 
{
    struct dirent *p;
    DIR *dir = opendir("/local");
    if (dir == NULL) {
        return -1;
    }
    while ((p = readdir(dir)) != NULL) {
        char *str = p->d_name;
        if ((strstr(str, ".bin") != NULL) || (strstr(str, ".BIN") != NULL)) {
            char buf[BUFSIZ];
            snprintf(buf, sizeof(buf) - 1, "/local/%s", str);
            if (remove(buf) == 0) {
                LOG("INFO: Deleted '%s'.\n", buf);
            } else {
                LOG("ERR : Delete '%s' failed.\n", buf);
            }
        }
    }
    closedir(dir);
    return 0;
}

 
/*
 * Output a message to a log file.
 *
 * @param format ...
 */
void FirmwareUpdater::LOG(const char* format, ...) {
    if (log) {
        FILE *fplog = fopen("/local/update.log", "a");
        if (fplog != NULL) {
            char buf[BUFSIZ];
            va_list p;
            va_start(p, format);
            vsnprintf(buf, sizeof(buf) - 1, format, p);
            fprintf(fplog, "%s", buf);
            va_end(p);
            fclose(fplog);
        }
    }


}
