/** file vs10xx.h
 * Headers for interfacing with the mp3 player chip.
 * Interfacing the New Way, not handling BSYNC -> not compatible with VS1001.
 */

#ifndef VS10XX_H
#define VS10XX_H

#include "mbed.h"

/** VS10xx SCI Write Command byte is 0x02 */
#define VS_WRITE_COMMAND 0x02
/** VS10xx SCI Read Command byte is 0x03 */
#define VS_READ_COMMAND  0x03

#define SPI_MODE	0x0   /**< VS10xx register */
#define SPI_STATUS	0x1   /**< VS10xx register */
#define SPI_BASS	0x2   /**< VS10xx register */
#define SPI_CLOCKF	0x3   /**< VS10xx register */
#define SPI_DECODE_TIME	0x4   /**< VS10xx register */
#define SPI_AUDATA	0x5   /**< VS10xx register */
#define SPI_WRAM	0x6   /**< VS10xx register */
#define SPI_WRAMADDR	0x7   /**< VS10xx register */
#define SPI_HDAT0	0x8   /**< VS10xx register */
#define SPI_HDAT1	0x9   /**< VS10xx register */
#define SPI_AIADDR	0xa   /**< VS10xx register */
#define SPI_VOL		0xb   /**< VS10xx register */
#define SPI_AICTRL0	0xc   /**< VS10xx register */
#define SPI_AICTRL1	0xd   /**< VS10xx register */
#define SPI_AICTRL2	0xe   /**< VS10xx register */
#define SPI_AICTRL3	0xf   /**< VS10xx register */

#define SM_DIFF           (1<< 0)
#define SM_LAYER12        (1<< 1) /* VS1063, VS1053, VS1033, VS1011 */
#define SM_RECORD_PATH    (1<< 1) /* VS1103 */
#define SM_RESET          (1<< 2)
#define SM_CANCEL         (1<< 3) /* VS1063, VS1053 */
#define SM_OUTOFWAV       (1<< 3) /* VS1033, VS1003, VS1011 */
#define SM_OUTOFMIDI      (1<< 3) /* VS1103 */
#define SM_EARSPEAKER_LO  (1<< 4) /* VS1053, VS1033 */
#define SM_PDOWN          (1<< 4) /* VS1003, VS1103 */
#define SM_TESTS          (1<< 5)
#define SM_STREAM         (1<< 6) /* VS1053, VS1033, VS1003, VS1011 */
#define SM_ICONF          (1<< 6) /* VS1103 */
#define SM_EARSPEAKER_HI  (1<< 7) /* VS1053, VS1033 */
#define SM_DACT           (1<< 8)
#define SM_SDIORD         (1<< 9)
#define SM_SDISHARE       (1<<10)
#define SM_SDINEW         (1<<11)
#define SM_ENCODE         (1<<12) /* VS1063 */
#define SM_ADPCM          (1<<12) /* VS1053, VS1033, VS1003 */
#define SM_EARSPEAKER1103 (1<<12) /* VS1103 */
#define SM_ADPCM_HP       (1<<13) /* VS1033, VS1003 */
#define SM_LINE1          (1<<14) /* VS1063, VS1053 */
#define SM_LINE_IN        (1<<14) /* VS1033, VS1003, VS1103 */
#define SM_CLK_RANGE      (1<<15) /* VS1063, VS1053, VS1033 */
#define SM_ADPCM_1103     (1<<15) /* VS1103 */

class vs10xx
{
public:
    vs10xx(PinName MOSI, PinName MISO, PinName SCLK, PinName XCS, 
           PinName XDCS,PinName DREQ, PinName XRESET);
    void writeRegister(unsigned char addressbyte, unsigned int value);
    unsigned int readRegister(unsigned char addressbyte); /** Read the 16-bit value of a VS10xx register */
    void writeData(unsigned char *databuf, unsigned char = 32); //write 32 bytes to vs1053
   
    void reset(void); //reset for vs10xx
    void softReset(void); /* Soft Reset of VS10xx (Between songs) */
    void setFreq(int freq);
    void setVolume(unsigned char vol);
    void loadPlugin(const unsigned short *plugin,int length);
private:
    SPI _spi;
    DigitalOut  _XCS;
    DigitalOut  _XDCS;  
    DigitalIn   _DREQ;
    DigitalOut  _XRESET;
      
};


const unsigned short recPlugin[40] = { /* Compressed plugin for recording*/
  0x0007, 0x0001, 0x8010, 0x0006, 0x001c, 0x3e12, 0xb817, 0x3e14, /*    0 */
  0xf812, 0x3e01, 0xb811, 0x0007, 0x9717, 0x0020, 0xffd2, 0x0030, /*    8 */
  0x11d1, 0x3111, 0x8024, 0x3704, 0xc024, 0x3b81, 0x8024, 0x3101, /*   10 */
  0x8024, 0x3b81, 0x8024, 0x3f04, 0xc024, 0x2808, 0x4800, 0x36f1, /*   18 */
  0x9811, 0x0007, 0x0001, 0x8028, 0x0006, 0x0002, 0x2a00, 0x040e,
};
   
//VS1053B: Patches with FLAC decoder 1.1 (2009-06-23) 
const unsigned short flacPlugin[5433] = 
{ 
   0x0007, 0x0001, 0x8300, 0x0006, 0x0202, 0xb080, 0x0024, 0x0007, /*    0 */ 
   0x9257, 0x3f00, 0x0024, 0x0030, 0x0297, 0x3f00, 0x0024, 0x0006, /*    8 */ 
   0x0017, 0x3f10, 0x0024, 0x3f00, 0x0024, 0x0000, 0xf8d7, 0xf400, /*   10 */ 
   0x55c0, 0x0000, 0x0817, 0xf400, 0x57c0, 0x0000, 0x004d, 0x000a, /*   18 */ 
   0x708f, 0x0000, 0xc44e, 0x280f, 0xe100, 0x0006, 0x2016, 0x0000, /*   20 */ 
   0x028d, 0x0014, 0x1b01, 0x2800, 0xc795, 0x0015, 0x59c0, 0x0000, /*   28 */ 
   0xfa0d, 0x0039, 0x324f, 0x0000, 0xd40e, 0x2920, 0x41c0, 0x0000, /*   30 */ 
   0x0024, 0x000a, 0x708f, 0x0000, 0xc44e, 0x280a, 0xcac0, 0x0000, /*   38 */ 
   0x028d, 0x6fc2, 0x0024, 0x0019, 0x9301, 0x2800, 0xcbd5, 0x0018, /*   40 */ 
   0x50c0, 0x000c, 0x0981, 0x0000, 0x18c2, 0x290c, 0x4840, 0x3613, /*   48 */ 
   0x0024, 0x290c, 0x4840, 0x4086, 0x184c, 0x6234, 0x0024, 0x0000, /*   50 */ 
   0x0024, 0x2800, 0xcb55, 0x0030, 0x0317, 0x3f00, 0x0024, 0x280a, /*   58 */ 
   0x71c0, 0x002c, 0x9d40, 0x6fc2, 0x0024, 0x0000, 0x0024, 0x2800, /*   60 */ 
   0xcb55, 0x0000, 0x0024, 0x2800, 0x9c40, 0x000a, 0xcac8, 0x3613, /*   68 */ 
   0x0024, 0x3e10, 0xb803, 0x3e14, 0x3811, 0x3e11, 0x3805, 0x3e00, /*   70 */ 
   0x3801, 0x0007, 0xc390, 0x0006, 0xa011, 0x3010, 0x0444, 0x3050, /*   78 */ 
   0x4405, 0x6458, 0x0302, 0xff94, 0x4081, 0x0003, 0xffc5, 0x48b6, /*   80 */ 
   0x0024, 0xff82, 0x0024, 0x42b2, 0x0042, 0xb458, 0x0003, 0x4cd6, /*   88 */ 
   0x9801, 0xf248, 0x1bc0, 0xb58a, 0x0024, 0x6de6, 0x1804, 0x0006, /*   90 */ 
   0x0010, 0x3810, 0x9bc5, 0x3800, 0xc024, 0x36f4, 0x1811, 0x36f0, /*   98 */ 
   0x9803, 0x283e, 0x2d80, 0x0fff, 0xffc3, 0x003e, 0x2d4f, 0x2800, /*   a0 */ 
   0xe580, 0x0000, 0xcd4e, 0x3413, 0x0024, 0x2800, 0xd605, 0xf400, /*   a8 */ 
   0x4510, 0x2800, 0xd9c0, 0x6894, 0x13cc, 0x3000, 0x184c, 0x6090, /*   b0 */ 
   0x93cc, 0x38b0, 0x3812, 0x3004, 0x4024, 0x0000, 0x0910, 0x3183, /*   b8 */ 
   0x0024, 0x3100, 0x4024, 0x6016, 0x0024, 0x000c, 0x8012, 0x2800, /*   c0 */ 
   0xd911, 0xb884, 0x104c, 0x6894, 0x3002, 0x2939, 0xb0c0, 0x3e10, /*   c8 */ 
   0x93cc, 0x4084, 0x9bd2, 0x4282, 0x0024, 0x0000, 0x0041, 0x2800, /*   d0 */ 
   0xdbc5, 0x6212, 0x0024, 0x0000, 0x0040, 0x2800, 0xe0c5, 0x000c, /*   d8 */ 
   0x8390, 0x2a00, 0xe440, 0x34c3, 0x0024, 0x3444, 0x0024, 0x3073, /*   e0 */ 
   0x0024, 0x3053, 0x0024, 0x3000, 0x0024, 0x6092, 0x098c, 0x0000, /*   e8 */ 
   0x0241, 0x2800, 0xe445, 0x32a0, 0x0024, 0x6012, 0x0024, 0x0000, /*   f0 */ 
   0x0024, 0x2800, 0xe455, 0x0000, 0x0024, 0x3613, 0x0024, 0x3001, /*   f8 */ 
   0x3844, 0x2920, 0x0580, 0x3009, 0x3852, 0xc090, 0x9bd2, 0x2800, /*  100 */ 
   0xe440, 0x3800, 0x1bc4, 0x000c, 0x4113, 0xb880, 0x2380, 0x3304, /*  108 */ 
   0x4024, 0x3800, 0x05cc, 0xcc92, 0x05cc, 0x3910, 0x0024, 0x3910, /*  110 */ 
   0x4024, 0x000c, 0x8110, 0x3910, 0x0024, 0x39f0, 0x4024, 0x3810, /*  118 */ 
   0x0024, 0x38d0, 0x4024, 0x3810, 0x0024, 0x38f0, 0x4024, 0x34c3, /*  120 */ 
   0x0024, 0x3444, 0x0024, 0x3073, 0x0024, 0x3063, 0x0024, 0x3000, /*  128 */ 
   0x0024, 0x4080, 0x0024, 0x0000, 0x0024, 0x2839, 0x53d5, 0x4284, /*  130 */ 
   0x0024, 0x3613, 0x0024, 0x2800, 0xe785, 0x6898, 0xb804, 0x0000, /*  138 */ 
   0x0084, 0x293b, 0x1cc0, 0x3613, 0x0024, 0x000c, 0x8117, 0x3711, /*  140 */ 
   0x0024, 0x37d1, 0x4024, 0x4e8a, 0x0024, 0x0000, 0x0015, 0x2800, /*  148 */ 
   0xea45, 0xce9a, 0x0024, 0x3f11, 0x0024, 0x3f01, 0x4024, 0x000c, /*  150 */ 
   0x8197, 0x408a, 0x9bc4, 0x3f15, 0x4024, 0x2800, 0xec85, 0x4284, /*  158 */ 
   0x3c15, 0x6590, 0x0024, 0x0000, 0x0024, 0x2839, 0x53d5, 0x4284, /*  160 */ 
   0x0024, 0x0000, 0x0024, 0x2800, 0xd4d8, 0x458a, 0x0024, 0x2a39, /*  168 */ 
   0x53c0, 0x3009, 0x3851, 0x3e14, 0xf812, 0x3e12, 0xb817, 0x0006, /*  170 */ 
   0xa057, 0x3e11, 0x9fd3, 0x0023, 0xffd2, 0x3e01, 0x0024, 0x0006, /*  178 */ 
   0x0011, 0x3111, 0x0024, 0x6498, 0x07c6, 0x868c, 0x2444, 0x3901, /*  180 */ 
   0x8e06, 0x0030, 0x0551, 0x3911, 0x8e06, 0x3961, 0x9c44, 0xf400, /*  188 */ 
   0x44c6, 0xd46c, 0x1bc4, 0x36f1, 0xbc13, 0x2800, 0xf815, 0x36f2, /*  190 */ 
   0x9817, 0x002b, 0xffd2, 0x3383, 0x188c, 0x3e01, 0x8c06, 0x0006, /*  198 */ 
   0xa097, 0x468c, 0xbc17, 0xf400, 0x4197, 0x2800, 0xf504, 0x3713, /*  1a0 */ 
   0x0024, 0x2800, 0xf545, 0x37e3, 0x0024, 0x3009, 0x2c17, 0x3383, /*  1a8 */ 
   0x0024, 0x3009, 0x0c06, 0x468c, 0x4197, 0x0006, 0xa052, 0x2800, /*  1b0 */ 
   0xf744, 0x3713, 0x2813, 0x2800, 0xf785, 0x37e3, 0x0024, 0x3009, /*  1b8 */ 
   0x2c17, 0x36f1, 0x8024, 0x36f2, 0x9817, 0x36f4, 0xd812, 0x2100, /*  1c0 */ 
   0x0000, 0x3904, 0x5bd1, 0x2a00, 0xed8e, 0x3e11, 0x7804, 0x0030, /*  1c8 */ 
   0x0257, 0x3701, 0x0024, 0x0013, 0x4d05, 0xd45b, 0xe0e1, 0x0007, /*  1d0 */ 
   0xc795, 0x2800, 0xff95, 0x0fff, 0xff45, 0x3511, 0x184c, 0x4488, /*  1d8 */ 
   0xb808, 0x0006, 0x8a97, 0x2800, 0xff45, 0x3009, 0x1c40, 0x3511, /*  1e0 */ 
   0x1fc1, 0x0000, 0x0020, 0xac52, 0x1405, 0x6ce2, 0x0024, 0x0000, /*  1e8 */ 
   0x0024, 0x2800, 0xff41, 0x68c2, 0x0024, 0x291a, 0x8a40, 0x3e10, /*  1f0 */ 
   0x0024, 0x2921, 0xca80, 0x3e00, 0x4024, 0x36f3, 0x0024, 0x3009, /*  1f8 */ 
   0x1bc8, 0x36f0, 0x1801, 0x2808, 0x9300, 0x3601, 0x5804, 0x0007, /*  200 */ 
   0x0001, 0x802e, 0x0006, 0x0002, 0x2800, 0xf900, 0x0007, 0x0001, /*  208 */ 
   0x8050, 0x0006, 0x0028, 0x3e12, 0x3800, 0x2911, 0xf140, 0x3e10, /*  210 */ 
   0x8024, 0xf400, 0x4595, 0x3593, 0x0024, 0x35f3, 0x0024, 0x3500, /*  218 */ 
   0x0024, 0x0021, 0x6d82, 0xd024, 0x44c0, 0x0006, 0xa402, 0x2800, /*  220 */ 
   0x1815, 0xd024, 0x0024, 0x0000, 0x0000, 0x2800, 0x1815, 0x000b, /*  228 */ 
   0x6d57, 0x3009, 0x3c00, 0x36f0, 0x8024, 0x36f2, 0x1800, 0x2000, /*  230 */ 
   0x0000, 0x0000, 0x0024, 0x0007, 0x0001, 0x8030, 0x0006, 0x0002, /*  238 */ 
   0x2800, 0x1400, 0x0007, 0x0001, 0x8064, 0x0006, 0x001c, 0x3e12, /*  240 */ 
   0xb817, 0x3e14, 0xf812, 0x3e01, 0xb811, 0x0007, 0x9717, 0x0020, /*  248 */ 
   0xffd2, 0x0030, 0x11d1, 0x3111, 0x8024, 0x3704, 0xc024, 0x3b81, /*  250 */ 
   0x8024, 0x3101, 0x8024, 0x3b81, 0x8024, 0x3f04, 0xc024, 0x2808, /*  258 */ 
   0x4800, 0x36f1, 0x9811, 0x0007, 0x0001, 0x8028, 0x0006, 0x0002, /*  260 */ 
   0x2a00, 0x190e, 0x0007, 0x0001, 0x8072, 0x0006, 0x0052, 0x3613, /*  268 */ 
   0x0024, 0x3e12, 0xb817, 0x3e12, 0x3815, 0x3e05, 0xb814, 0x3635, /*  270 */ 
   0x0024, 0x0000, 0x800a, 0x3e10, 0x7802, 0x3e14, 0x0024, 0x2900, /*  278 */ 
   0xb680, 0x0000, 0x0201, 0x0000, 0x0601, 0x3413, 0x184c, 0x2902, /*  280 */ 
   0x4700, 0x3cf0, 0x0024, 0x3413, 0x184c, 0x3400, 0x3040, 0x3009, /*  288 */ 
   0x33c1, 0x0000, 0x1fc1, 0xb010, 0x0024, 0x6014, 0x9040, 0x0006, /*  290 */ 
   0x0bd0, 0x2800, 0x2355, 0x0000, 0x0024, 0x34e3, 0x1bcc, 0x6890, /*  298 */ 
   0x0024, 0x2800, 0x2500, 0xb880, 0x2000, 0x3e10, 0x1381, 0x2902, /*  2a0 */ 
   0x8640, 0x3e00, 0x4024, 0x003f, 0xfe41, 0x36e3, 0x104c, 0x34f0, /*  2a8 */ 
   0x0024, 0xa010, 0x0024, 0x36f4, 0x0024, 0x36f0, 0x5802, 0x3405, /*  2b0 */ 
   0x9014, 0x36f3, 0x0024, 0x36f2, 0x1815, 0x2000, 0x0000, 0x36f2, /*  2b8 */ 
   0x9817, 0x0007, 0x0001, 0x809b, 0x0006, 0x004a, 0x3613, 0x0024, /*  2c0 */ 
   0x3e12, 0xb817, 0x3e12, 0x3815, 0x3e05, 0xb814, 0x3615, 0x0024, /*  2c8 */ 
   0x0000, 0x800a, 0x3e10, 0xb804, 0x3e01, 0x534c, 0xbe8a, 0x10c0, /*  2d0 */ 
   0x4080, 0x0024, 0x0000, 0x0024, 0x2800, 0x2e45, 0x0000, 0x0024, /*  2d8 */ 
   0x2902, 0x4700, 0x4082, 0x184c, 0x4c8a, 0x134c, 0x0000, 0x0001, /*  2e0 */ 
   0x6890, 0x10c2, 0x4294, 0x0024, 0xac22, 0x0024, 0xbec2, 0x0024, /*  2e8 */ 
   0x0000, 0x0024, 0x2800, 0x2e45, 0x0000, 0x0024, 0x6890, 0x134c, /*  2f0 */ 
   0xb882, 0x10c2, 0xac22, 0x0024, 0x4c92, 0x0024, 0xdc92, 0x0024, /*  2f8 */ 
   0xceca, 0x0024, 0x4e82, 0x1bc5, 0x36f0, 0x9804, 0x3405, 0x9014, /*  300 */ 
   0x36f3, 0x0024, 0x36f2, 0x1815, 0x2000, 0x0000, 0x36f2, 0x9817, /*  308 */ 
   0x0007, 0x0001, 0x80c0, 0x0006, 0x00e4, 0x3613, 0x0024, 0x3e12, /*  310 */ 
   0xb817, 0x3e12, 0x3815, 0x3e05, 0xb814, 0x3645, 0x0024, 0x0000, /*  318 */ 
   0x800a, 0x3e10, 0x3801, 0x3e10, 0xb803, 0x3e11, 0x3805, 0x3e11, /*  320 */ 
   0xb807, 0x3e14, 0x104c, 0x2900, 0xb680, 0x0000, 0x0081, 0x4080, /*  328 */ 
   0x3040, 0x0000, 0x0101, 0x2800, 0x3585, 0x0000, 0x0024, 0x4090, /*  330 */ 
   0x138c, 0x0006, 0x0c10, 0x2800, 0x4995, 0x0000, 0x0024, 0x3423, /*  338 */ 
   0x0024, 0x2900, 0xb680, 0x3613, 0x0024, 0xb880, 0x3000, 0x2800, /*  340 */ 
   0x4740, 0x3009, 0x3380, 0x2900, 0xb680, 0x4122, 0x10cc, 0x3cf0, /*  348 */ 
   0x0024, 0x3001, 0x0024, 0x3400, 0x0024, 0x6800, 0x0024, 0xa408, /*  350 */ 
   0x9040, 0x4080, 0x0024, 0x0000, 0x07c1, 0x2800, 0x3b15, 0x6894, /*  358 */ 
   0x1380, 0x6894, 0x130c, 0x3460, 0x0024, 0x6408, 0x4481, 0x4102, /*  360 */ 
   0x1380, 0xf400, 0x4052, 0x0000, 0x07c1, 0x34f0, 0xc024, 0x6234, /*  368 */ 
   0x0024, 0x6824, 0x0024, 0xa122, 0x0024, 0x6014, 0x0024, 0x0000, /*  370 */ 
   0x0141, 0x2800, 0x4215, 0x0000, 0x0024, 0x2900, 0xb680, 0x3613, /*  378 */ 
   0x0024, 0x2800, 0x4080, 0xb88a, 0x4002, 0x2900, 0x26c0, 0x3e00, /*  380 */ 
   0x8024, 0x4c8e, 0xa801, 0x0000, 0x0201, 0x3a10, 0x1bcc, 0x3000, /*  388 */ 
   0x0024, 0xb010, 0x0024, 0x0000, 0x0024, 0x2800, 0x4615, 0x659a, /*  390 */ 
   0x0024, 0x6540, 0x184c, 0x0030, 0x0010, 0x2800, 0x3e08, 0x0000, /*  398 */ 
   0x0024, 0x2800, 0x4600, 0x36f3, 0x0024, 0x2800, 0x44c0, 0xb88a, /*  3a0 */ 
   0x0024, 0x2902, 0x0e80, 0x34d0, 0x4024, 0x4c8f, 0xa0a1, 0x0000, /*  3a8 */ 
   0x0201, 0x3000, 0x084c, 0xb010, 0x0024, 0x0000, 0x0024, 0x2800, /*  3b0 */ 
   0x4615, 0x659a, 0x0024, 0x6540, 0x10cc, 0x0030, 0x0010, 0x2800, /*  3b8 */ 
   0x4288, 0x0000, 0x0024, 0x34d3, 0x0024, 0x3423, 0x0024, 0xf400, /*  3c0 */ 
   0x4510, 0x3009, 0x1380, 0x6090, 0x0024, 0x3009, 0x2000, 0x6892, /*  3c8 */ 
   0x108c, 0x34f0, 0x9000, 0xa122, 0x984c, 0x6016, 0x13c1, 0x0000, /*  3d0 */ 
   0x0102, 0x2800, 0x36c8, 0x0006, 0x0d10, 0x2800, 0x4a00, 0x3009, /*  3d8 */ 
   0x1bcc, 0x6890, 0x0024, 0x3800, 0x0024, 0x36f4, 0x0024, 0x36f1, /*  3e0 */ 
   0x9807, 0x36f1, 0x1805, 0x36f0, 0x9803, 0x36f0, 0x1801, 0x3405, /*  3e8 */ 
   0x9014, 0x36f3, 0x0024, 0x36f2, 0x1815, 0x2000, 0x0000, 0x36f2, /*  3f0 */ 
   0x9817, 0x0007, 0x0001, 0x8132, 0x0006, 0x0048, 0x3613, 0x0024, /*  3f8 */ 
   0x3e12, 0xb817, 0x3e12, 0x3815, 0x3e05, 0xb814, 0x3615, 0x0024, /*  400 */ 
   0x0000, 0x800a, 0x3e10, 0x3801, 0x3e10, 0xb804, 0x3e11, 0xb807, /*  408 */ 
   0x3e14, 0x3811, 0x3e04, 0x934c, 0x3430, 0x0024, 0x4080, 0x0024, /*  410 */ 
   0x0000, 0x0206, 0x2800, 0x5305, 0x0006, 0x0d11, 0x3101, 0x130c, /*  418 */ 
   0xff0c, 0x1102, 0x6408, 0x0024, 0x4204, 0x0024, 0xb882, 0x4092, /*  420 */ 
   0x1006, 0x0e02, 0x48be, 0x0024, 0x4264, 0x0024, 0x2901, 0xfa00, /*  428 */ 
   0xf400, 0x4090, 0x36f4, 0x8024, 0x36f4, 0x1811, 0x36f1, 0x9807, /*  430 */ 
   0x36f0, 0x9804, 0x36f0, 0x1801, 0x3405, 0x9014, 0x36f3, 0x0024, /*  438 */ 
   0x36f2, 0x1815, 0x2000, 0x0000, 0x36f2, 0x9817, 0x0007, 0x0001, /*  440 */ 
   0x1802, 0x0006, 0x0002, 0x0194, 0x01af, 0x0006, 0x8006, 0x01f6, /*  448 */ 
   0x0006, 0x0025, 0x01cb, 0x01cb, 0x01cb, 0x01cb, 0x01cb, 0x0581, /*  450 */ 
   0x0565, 0x0569, 0x056d, 0x0571, 0x0575, 0x0579, 0x057d, 0x05b3, /*  458 */ 
   0x05b7, 0x05ba, 0x05ba, 0x05ba, 0x05ba, 0x05c2, 0x05d6, 0x06a5, /*  460 */ 
   0x060e, 0x0613, 0x0619, 0x061f, 0x0624, 0x0629, 0x062e, 0x0633, /*  468 */ 
   0x0638, 0x063d, 0x0642, 0x0647, 0x0661, 0x0681, 0x06a1, 0x0007, /*  470 */ 
   0x0001, 0x8156, 0x0006, 0x0236, 0x3613, 0x0024, 0x3e12, 0xb817, /*  478 */ 
   0x3e12, 0x3815, 0x3e05, 0xb814, 0x3675, 0x0024, 0x3633, 0x0024, /*  480 */ 
   0x0000, 0x800a, 0x3e10, 0x3801, 0x0000, 0x0181, 0x3e10, 0xb803, /*  488 */ 
   0x3e11, 0x3806, 0x3e11, 0xf810, 0x3e14, 0x7812, 0x3e13, 0xf80e, /*  490 */ 
   0x2902, 0x1840, 0x3e03, 0x4024, 0x2900, 0xb680, 0x4088, 0x184c, /*  498 */ 
   0x3413, 0x184c, 0x2900, 0xb680, 0x6892, 0x33c0, 0x3423, 0x0024, /*  4a0 */ 
   0x4080, 0x3040, 0x0000, 0x0000, 0x2800, 0x60c5, 0x0000, 0x0024, /*  4a8 */ 
   0x6890, 0x0024, 0x2902, 0x1840, 0x3cd0, 0x0024, 0x4080, 0x0024, /*  4b0 */ 
   0x0000, 0x0024, 0x2800, 0x6115, 0x0000, 0x0024, 0x3433, 0x0024, /*  4b8 */ 
   0xf400, 0x4510, 0x34d0, 0x0024, 0x6090, 0x0024, 0x2902, 0x1840, /*  4c0 */ 
   0x3800, 0x0024, 0x4080, 0x10cc, 0xf400, 0x4510, 0x2800, 0x5e85, /*  4c8 */ 
   0x34d0, 0x0024, 0x2800, 0x6100, 0x0000, 0x0024, 0x3cd0, 0x0024, /*  4d0 */ 
   0x3433, 0x0024, 0x34a0, 0x0024, 0xf400, 0x4510, 0x3430, 0x4024, /*  4d8 */ 
   0x6100, 0x0024, 0x0000, 0x0341, 0x3840, 0x0024, 0x3000, 0x0024, /*  4e0 */ 
   0x6012, 0x0024, 0x0006, 0x0081, 0x2800, 0x7d81, 0x4012, 0x0024, /*  4e8 */ 
   0xf400, 0x4057, 0x3702, 0x0024, 0x2000, 0x0000, 0x0000, 0x0024, /*  4f0 */ 
   0x34d3, 0x184c, 0x3430, 0x8024, 0x2900, 0x26c0, 0x3e00, 0x8024, /*  4f8 */ 
   0x36f3, 0x11cc, 0xb888, 0x104c, 0x3c10, 0x0024, 0x3c90, 0x4024, /*  500 */ 
   0x2800, 0x69c0, 0x34e3, 0x0024, 0x4f82, 0x128c, 0x3400, 0x4024, /*  508 */ 
   0x4142, 0x0024, 0xf400, 0x4050, 0x3800, 0x0024, 0x3440, 0x4024, /*  510 */ 
   0x4142, 0x0024, 0x6498, 0x4050, 0x3009, 0x2007, 0x0006, 0x0d10, /*  518 */ 
   0x3000, 0x11cc, 0x6402, 0x104c, 0x3411, 0x8024, 0x2800, 0x6788, /*  520 */ 
   0x3491, 0xc024, 0x2800, 0x9940, 0x34e3, 0x0024, 0x2800, 0x70c0, /*  528 */ 
   0xb888, 0x0024, 0x2900, 0x26c0, 0x3e00, 0x8024, 0x4c8e, 0x130c, /*  530 */ 
   0x3400, 0x5bcc, 0x4142, 0x0024, 0xf400, 0x4050, 0x3800, 0x0024, /*  538 */ 
   0x3440, 0x4024, 0x4142, 0x0024, 0xf400, 0x4050, 0x0000, 0x0201, /*  540 */ 
   0x3009, 0x2007, 0x0030, 0x0010, 0x3000, 0x0024, 0xb010, 0x0024, /*  548 */ 
   0x0000, 0x0024, 0x2800, 0x9955, 0x6498, 0x0024, 0x0006, 0x0d10, /*  550 */ 
   0x3000, 0x134c, 0x6402, 0x10c2, 0x0000, 0x0024, 0x2800, 0x6c48, /*  558 */ 
   0x3009, 0x184c, 0x2800, 0x9940, 0x3009, 0x1bcc, 0x0000, 0x0201, /*  560 */ 
   0xb888, 0x104c, 0x3430, 0x184c, 0x6010, 0x0024, 0x6402, 0x3000, /*  568 */ 
   0x0000, 0x0201, 0x2800, 0x7958, 0x0030, 0x0010, 0x4090, 0x124c, /*  570 */ 
   0x2400, 0x7840, 0x0000, 0x0024, 0x3430, 0x8024, 0x2900, 0x26c0, /*  578 */ 
   0x3e00, 0x8024, 0x4c8e, 0x130c, 0x3400, 0x4024, 0x4142, 0x0024, /*  580 */ 
   0xf400, 0x4050, 0x3800, 0x0024, 0x3410, 0x4024, 0x4142, 0x0024, /*  588 */ 
   0x6498, 0x4050, 0x3009, 0x2007, 0x0030, 0x0010, 0x0000, 0x0201, /*  590 */ 
   0x3473, 0x0024, 0x3490, 0x0024, 0x3e00, 0x13cc, 0x2900, 0x3000, /*  598 */ 
   0x3444, 0x8024, 0x3000, 0x1bcc, 0xb010, 0x0024, 0x0000, 0x0024, /*  5a0 */ 
   0x2800, 0x9955, 0x0000, 0x0024, 0x34c3, 0x184c, 0x3470, 0x0024, /*  5a8 */ 
   0x3e10, 0x104c, 0x34c0, 0x4024, 0x2900, 0x4c80, 0x3e00, 0x4024, /*  5b0 */ 
   0x2800, 0x9940, 0x36e3, 0x0024, 0x0000, 0x0801, 0x3413, 0x0024, /*  5b8 */ 
   0x34f0, 0x0024, 0x6012, 0x0024, 0x0000, 0x07c1, 0x2800, 0x9888, /*  5c0 */ 
   0x0000, 0x0024, 0x6010, 0x114c, 0xb888, 0x32c0, 0x6402, 0x0024, /*  5c8 */ 
   0x0000, 0x0101, 0x2800, 0x8518, 0x0000, 0x0024, 0x4090, 0x134c, /*  5d0 */ 
   0x2400, 0x8440, 0x3009, 0x184c, 0x3430, 0x8024, 0x2900, 0x26c0, /*  5d8 */ 
   0x3e00, 0x8024, 0x4c8e, 0x130c, 0x3400, 0x4024, 0x4142, 0x0024, /*  5e0 */ 
   0xf400, 0x4050, 0x3800, 0x0024, 0x3410, 0x4024, 0x4142, 0x0024, /*  5e8 */ 
   0x6498, 0x4050, 0x3009, 0x2007, 0x0000, 0x0101, 0x3433, 0x1bcc, /*  5f0 */ 
   0x2900, 0xb680, 0x3613, 0x0024, 0x0000, 0x0141, 0x6090, 0x118c, /*  5f8 */ 
   0x2900, 0xb680, 0x3ca0, 0x184c, 0x3473, 0x184c, 0xb888, 0x3380, /*  600 */ 
   0x3400, 0x0024, 0x6402, 0x0024, 0x0000, 0x0201, 0x2800, 0x8bd8, /*  608 */ 
   0x0000, 0x0024, 0x4090, 0x104c, 0x2400, 0x8b00, 0x0000, 0x0024, /*  610 */ 
   0x34a0, 0x8024, 0x2900, 0x26c0, 0x3e00, 0x8024, 0x0006, 0x0002, /*  618 */ 
   0x4244, 0x118c, 0x4244, 0x0024, 0x6498, 0x4095, 0x3009, 0x3440, /*  620 */ 
   0x3009, 0x37c1, 0x0000, 0x0201, 0x34f3, 0x0024, 0x0030, 0x0010, /*  628 */ 
   0x3490, 0x0024, 0x3e00, 0x138c, 0x2900, 0x3000, 0x3444, 0x8024, /*  630 */ 
   0x3000, 0x1bcc, 0xb010, 0x0024, 0x0000, 0x0024, 0x2800, 0x9955, /*  638 */ 
   0x4112, 0x0024, 0x3463, 0x0024, 0x34a0, 0x0024, 0x6012, 0x0024, /*  640 */ 
   0x0006, 0x0cd1, 0x2800, 0x9519, 0x0000, 0x0024, 0x3100, 0x11cc, /*  648 */ 
   0x3490, 0x4024, 0x4010, 0x0024, 0x0000, 0x0a01, 0x6012, 0x0024, /*  650 */ 
   0x0006, 0x0d11, 0x2800, 0x9518, 0x0000, 0x0024, 0x3613, 0x114c, /*  658 */ 
   0x3101, 0x3804, 0x3490, 0x8024, 0x6428, 0x138c, 0x3470, 0x8024, /*  660 */ 
   0x3423, 0x0024, 0x3420, 0xc024, 0x4234, 0x1241, 0x4380, 0x4092, /*  668 */ 
   0x2901, 0xfa00, 0x0006, 0x0010, 0x2800, 0x9940, 0x3009, 0x1bcc, /*  670 */ 
   0x0006, 0x0d11, 0x3613, 0x114c, 0x3101, 0x3804, 0x3490, 0x8024, /*  678 */ 
   0x6428, 0x138c, 0x3470, 0x8024, 0x3423, 0x0024, 0x3420, 0xc024, /*  680 */ 
   0x4234, 0x1241, 0x4380, 0x4092, 0x2902, 0x03c0, 0x0006, 0x0010, /*  688 */ 
   0x2800, 0x9940, 0x3009, 0x1bcc, 0x0006, 0x0c10, 0x6890, 0x0024, /*  690 */ 
   0x3800, 0x0024, 0x36f3, 0x4024, 0x36f3, 0xd80e, 0x36f4, 0x5812, /*  698 */ 
   0x36f1, 0xd810, 0x36f1, 0x1806, 0x36f0, 0x9803, 0x36f0, 0x1801, /*  6a0 */ 
   0x3405, 0x9014, 0x36f3, 0x0024, 0x36f2, 0x1815, 0x2000, 0x0000, /*  6a8 */ 
   0x36f2, 0x9817, 0x0007, 0x0001, 0x8401, 0x0006, 0x016e, 0x3613, /*  6b0 */ 
   0x0024, 0x3e12, 0xb817, 0x3e12, 0x3815, 0x3e05, 0xb814, 0x3615, /*  6b8 */ 
   0x0024, 0x0000, 0x800a, 0x3e10, 0x7802, 0x3e10, 0xf804, 0x0000, /*  6c0 */ 
   0x3fc3, 0x3e11, 0x7806, 0x3e11, 0xf810, 0xbc82, 0x12cc, 0x3404, /*  6c8 */ 
   0x0024, 0x3023, 0x0024, 0x3810, 0x0024, 0x38f0, 0x4024, 0x3454, /*  6d0 */ 
   0x0024, 0x3810, 0x0024, 0x38f0, 0x4024, 0x2900, 0xb680, 0x0000, /*  6d8 */ 
   0x0201, 0x0006, 0x8081, 0x4088, 0x134c, 0x3400, 0x8024, 0xd204, /*  6e0 */ 
   0x0024, 0xb234, 0x0024, 0x4122, 0x0024, 0xf400, 0x4055, 0x3500, /*  6e8 */ 
   0x0024, 0x3c30, 0x0024, 0x0000, 0x2000, 0xb400, 0x0024, 0x0000, /*  6f0 */ 
   0x3001, 0x2801, 0x0ad5, 0x0000, 0x3800, 0x0000, 0x0041, 0xfe42, /*  6f8 */ 
   0x12cc, 0x48b2, 0x1090, 0x3810, 0x0024, 0x38f0, 0x4024, 0x2801, /*  700 */ 
   0x2bc0, 0x3430, 0x0024, 0xb400, 0x0024, 0x6012, 0x0024, 0x0000, /*  708 */ 
   0x3801, 0x2801, 0x0e15, 0x0000, 0x3c00, 0x0000, 0x07c0, 0x0000, /*  710 */ 
   0x0041, 0xb400, 0x12cc, 0xfe02, 0x1150, 0x48b2, 0x0024, 0x689a, /*  718 */ 
   0x2040, 0x2801, 0x2a40, 0x38f0, 0x4024, 0xb400, 0x0024, 0x6012, /*  720 */ 
   0x0024, 0x0000, 0x3c01, 0x2801, 0x1195, 0x0000, 0x3e00, 0x0000, /*  728 */ 
   0x03c0, 0x0000, 0x0085, 0x4592, 0x12cc, 0xb400, 0x1150, 0xfe02, /*  730 */ 
   0x0024, 0x48b2, 0x0024, 0x3810, 0x0024, 0x2801, 0x2a40, 0x38f0, /*  738 */ 
   0x4024, 0xb400, 0x0024, 0x6012, 0x0024, 0x0000, 0x3e01, 0x2801, /*  740 */ 
   0x1515, 0x0000, 0x3f00, 0x0000, 0x01c0, 0xf20a, 0x12cc, 0xb400, /*  748 */ 
   0x1150, 0xf252, 0x0024, 0xfe02, 0x0024, 0x48b2, 0x0024, 0x3810, /*  750 */ 
   0x0024, 0x2801, 0x2a40, 0x38f0, 0x4024, 0xb400, 0x130c, 0x6012, /*  758 */ 
   0x0024, 0x0000, 0x3f01, 0x2801, 0x1895, 0x4390, 0x0024, 0x0000, /*  760 */ 
   0x0041, 0x0000, 0x0105, 0x4590, 0x13cc, 0xb400, 0x1150, 0xfe02, /*  768 */ 
   0x0024, 0x48b2, 0x0024, 0x3810, 0x0024, 0x2801, 0x2a40, 0x38f0, /*  770 */ 
   0x4024, 0xb400, 0x0024, 0x6012, 0x1100, 0x0000, 0x01c1, 0x2801, /*  778 */ 
   0x1c15, 0x0000, 0x0024, 0x0000, 0x0041, 0x0000, 0x0145, 0x6890, /*  780 */ 
   0x12cc, 0xb400, 0x1150, 0xfe02, 0x0024, 0x48b2, 0x0024, 0x3810, /*  788 */ 
   0x0024, 0x2801, 0x2a40, 0x38f0, 0x4024, 0x6012, 0x0024, 0x0000, /*  790 */ 
   0x3f81, 0x2801, 0x1e95, 0xb430, 0x0024, 0x6012, 0x0024, 0x0000, /*  798 */ 
   0x0024, 0x2801, 0x1e95, 0x0000, 0x0024, 0x2801, 0x2a40, 0x0000, /*  7a0 */ 
   0x0185, 0x2801, 0x2bc0, 0xc890, 0x0024, 0x0000, 0x3fc3, 0x0000, /*  7a8 */ 
   0x0201, 0x2900, 0xb680, 0x3613, 0x0024, 0x0006, 0x8081, 0x4088, /*  7b0 */ 
   0x134c, 0x3400, 0x8024, 0xd204, 0x0024, 0xb234, 0x0024, 0x4122, /*  7b8 */ 
   0x0024, 0xf400, 0x4055, 0x0000, 0x2001, 0x3500, 0x0024, 0x3c30, /*  7c0 */ 
   0x0024, 0x0000, 0x3000, 0xb400, 0x0024, 0x6012, 0x0024, 0x0000, /*  7c8 */ 
   0x0182, 0x2801, 0x2485, 0x0000, 0x0024, 0x2801, 0x2bc0, 0xc890, /*  7d0 */ 
   0x0024, 0x459a, 0x12cc, 0x3404, 0x0024, 0x3023, 0x0024, 0x3010, /*  7d8 */ 
   0x0024, 0x30d0, 0x4024, 0xac22, 0x0046, 0x003f, 0xf982, 0x3011, /*  7e0 */ 
   0xc024, 0x0000, 0x0023, 0xaf2e, 0x0024, 0x0000, 0x0182, 0xccf2, /*  7e8 */ 
   0x0024, 0x0000, 0x0fc6, 0x0000, 0x0047, 0xb46c, 0x2040, 0xfe6e, /*  7f0 */ 
   0x23c1, 0x3454, 0x0024, 0x3010, 0x0024, 0x30f0, 0x4024, 0xac22, /*  7f8 */ 
   0x0024, 0xccb2, 0x0024, 0x3810, 0x0024, 0x38f0, 0x4024, 0x458a, /*  800 */ 
   0x0024, 0x0000, 0x0201, 0x2801, 0x1f95, 0x0000, 0x3fc3, 0x34d3, /*  808 */ 
   0x0024, 0x3430, 0x0024, 0x36f1, 0xd810, 0x36f1, 0x5806, 0x36f0, /*  810 */ 
   0xd804, 0x36f0, 0x5802, 0x3405, 0x9014, 0x36f3, 0x0024, 0x36f2, /*  818 */ 
   0x1815, 0x2000, 0x0000, 0x36f2, 0x9817, 0x0007, 0x0001, 0x84b8, /*  820 */ 
   0x0006, 0x0660, 0x3613, 0x0024, 0x3e12, 0xb817, 0x0000, 0x0417, /*  828 */ 
   0x3e12, 0x3815, 0x3e05, 0xb814, 0x3685, 0x0024, 0x0000, 0x800a, /*  830 */ 
   0x3e10, 0x3801, 0x3e10, 0xb803, 0x3e11, 0x3805, 0x3e11, 0xb807, /*  838 */ 
   0x3e14, 0x3811, 0x3e14, 0xb813, 0x3e13, 0xf80e, 0x3e03, 0x4024, /*  840 */ 
   0x2902, 0x7980, 0x0000, 0x0381, 0x000f, 0xff81, 0x6012, 0x0024, /*  848 */ 
   0x0000, 0x0201, 0x2801, 0x3445, 0x0000, 0x0024, 0x3613, 0x0024, /*  850 */ 
   0x2900, 0xb680, 0x0001, 0xf6c8, 0x0006, 0x8a10, 0xb880, 0x184c, /*  858 */ 
   0x2902, 0x57c0, 0x3e00, 0x11cc, 0x3473, 0x0024, 0x3c90, 0x0024, /*  860 */ 
   0x3493, 0x0024, 0x2900, 0xb680, 0x0000, 0x0401, 0x3473, 0x184c, /*  868 */ 
   0x3423, 0x0024, 0x3c50, 0x0000, 0x4080, 0x0024, 0x0006, 0x8081, /*  870 */ 
   0x2801, 0x3ac5, 0x003f, 0xfe04, 0x34b3, 0x0024, 0x3490, 0x8024, /*  878 */ 
   0xa244, 0x138c, 0x2902, 0x52c0, 0xb880, 0x0024, 0x4082, 0x184c, /*  880 */ 
   0x0006, 0x8081, 0x2902, 0x57c0, 0x3e00, 0x11cc, 0x3473, 0x0024, /*  888 */ 
   0x0006, 0x0c51, 0x34b0, 0x8024, 0x3400, 0xc024, 0xa346, 0x0024, /*  890 */ 
   0xd234, 0x0024, 0x0000, 0x3fc3, 0xb234, 0x0024, 0x4122, 0x1142, /*  898 */ 
   0xf400, 0x4055, 0x0006, 0x8081, 0x3500, 0x0024, 0xd024, 0x3000, /*  8a0 */ 
   0xb234, 0x0024, 0x4122, 0x0024, 0xf400, 0x4055, 0x0000, 0x0041, /*  8a8 */ 
   0x3500, 0x0024, 0x3cb0, 0x0024, 0x3490, 0x0024, 0xf100, 0x13cc, /*  8b0 */ 
   0xf100, 0x0024, 0xfe02, 0x0024, 0x48b2, 0x0024, 0x3c10, 0x0024, /*  8b8 */ 
   0x6892, 0x31c1, 0x34a0, 0x0024, 0xf100, 0x0024, 0xb010, 0x0024, /*  8c0 */ 
   0x3c60, 0x0024, 0x34b0, 0x0024, 0xb010, 0x0024, 0x0000, 0x0201, /*  8c8 */ 
   0x2900, 0xb680, 0x3cc0, 0x0024, 0x0006, 0x8081, 0x3473, 0x184c, /*  8d0 */ 
   0x3423, 0x0024, 0x3c50, 0x0024, 0x34b0, 0x8024, 0x3450, 0xc024, /*  8d8 */ 
   0xd234, 0x0024, 0x0000, 0x3fc3, 0xb234, 0x0024, 0x4122, 0x0024, /*  8e0 */ 
   0xf400, 0x4055, 0x003f, 0xff01, 0x3500, 0x0024, 0x3cb0, 0x0024, /*  8e8 */ 
   0x34c0, 0x0024, 0xa010, 0x0024, 0x0000, 0x03c1, 0x3c40, 0x0024, /*  8f0 */ 
   0x34d0, 0x0024, 0xb010, 0x0024, 0x0000, 0x0201, 0x2900, 0xb680, /*  8f8 */ 
   0x3ca0, 0x0024, 0x0006, 0x8081, 0x3473, 0x0024, 0x3423, 0x0024, /*  900 */ 
   0x3c50, 0x0024, 0x34b0, 0x8024, 0x3450, 0xc024, 0xd234, 0x0024, /*  908 */ 
   0x0000, 0x3fc3, 0xb234, 0x0024, 0x4122, 0x0024, 0xf400, 0x4055, /*  910 */ 
   0x003f, 0xff01, 0x3500, 0x0024, 0x3cb0, 0x0024, 0x3400, 0x0024, /*  918 */ 
   0xa010, 0x0024, 0x0000, 0x01c1, 0x3900, 0x0024, 0x34e0, 0x0024, /*  920 */ 
   0xf100, 0x0024, 0xb010, 0x0024, 0x6892, 0x3080, 0x34f0, 0x0024, /*  928 */ 
   0xb010, 0x0024, 0x3cb0, 0x0024, 0x3450, 0x0024, 0x3490, 0x4024, /*  930 */ 
   0xc010, 0x93cc, 0x0000, 0x0181, 0x2801, 0x5185, 0x3100, 0x0024, /*  938 */ 
   0x6890, 0x07cc, 0x2801, 0xf6c0, 0x3900, 0x0024, 0x6012, 0x0024, /*  940 */ 
   0x0000, 0x0201, 0x2801, 0x5318, 0x0000, 0x0024, 0x2801, 0x5600, /*  948 */ 
   0x6090, 0x044c, 0x6012, 0x0024, 0x0000, 0x0281, 0x2801, 0x5548, /*  950 */ 
   0x6012, 0x0024, 0x0000, 0x0080, 0x2801, 0x5559, 0x0000, 0x0024, /*  958 */ 
   0x2801, 0x5600, 0x3113, 0x0024, 0x6890, 0x07cc, 0x2801, 0xf6c0, /*  960 */ 
   0x3900, 0x0024, 0x0000, 0x0201, 0x3900, 0x11cc, 0x3490, 0x0024, /*  968 */ 
   0x6012, 0x0024, 0x0006, 0x03c1, 0x2801, 0x6041, 0x4012, 0x0024, /*  970 */ 
   0xf400, 0x4057, 0x3702, 0x0024, 0x2000, 0x0000, 0x0000, 0x0024, /*  978 */ 
   0x2801, 0x6040, 0x0000, 0x0024, 0x0000, 0x0200, 0x0006, 0x0cd0, /*  980 */ 
   0x2801, 0x6040, 0x3800, 0x0024, 0x0000, 0x0300, 0x0006, 0x0cd0, /*  988 */ 
   0x2801, 0x6040, 0x3800, 0x0024, 0x0006, 0x0c10, 0x6890, 0x0024, /*  990 */ 
   0x2801, 0xf6c0, 0x3800, 0x0024, 0x0000, 0x0400, 0x0006, 0x0cd0, /*  998 */ 
   0x2801, 0x6040, 0x3800, 0x0024, 0x0000, 0x0500, 0x0006, 0x0cd0, /*  9a0 */ 
   0x2801, 0x6040, 0x3800, 0x0024, 0x0000, 0x0600, 0x0006, 0x0cd0, /*  9a8 */ 
   0x2801, 0x6040, 0x3800, 0x0024, 0x0006, 0x0c10, 0x6890, 0x0024, /*  9b0 */ 
   0x2801, 0xf6c0, 0x3800, 0x0024, 0x0000, 0x0281, 0x3443, 0x184c, /*  9b8 */ 
   0x34c0, 0x0024, 0x4080, 0x0024, 0xf400, 0x4500, 0x2801, 0x6605, /*  9c0 */ 
   0x4010, 0x91cc, 0x3e10, 0x11cc, 0x0000, 0x01c0, 0x3e10, 0x0024, /*  9c8 */ 
   0x3490, 0x0024, 0x2901, 0x0040, 0x3e00, 0x124c, 0x36d3, 0x11cc, /*  9d0 */ 
   0x3473, 0x0024, 0x4080, 0x3240, 0x3493, 0x0024, 0x2801, 0x6a18, /*  9d8 */ 
   0x0000, 0x0024, 0x0006, 0x0bd0, 0x6890, 0x0024, 0x2801, 0xf6c0, /*  9e0 */ 
   0x3800, 0x0024, 0x3e10, 0x11cc, 0x0000, 0x0180, 0x3e10, 0x0024, /*  9e8 */ 
   0x3490, 0x0024, 0x2901, 0x0040, 0x3e00, 0x124c, 0x36d3, 0x11cc, /*  9f0 */ 
   0x3473, 0x0024, 0x4080, 0x3240, 0x3493, 0x0024, 0x2801, 0x6a18, /*  9f8 */ 
   0x0000, 0x0024, 0x0006, 0x0bd0, 0x6890, 0x0024, 0x2801, 0xf6c0, /*  a00 */ 
   0x3800, 0x0024, 0x0000, 0x0201, 0x3453, 0x0024, 0x34b0, 0x0024, /*  a08 */ 
   0x6012, 0x0024, 0x0006, 0x05c1, 0x2801, 0x7c81, 0x4012, 0x0024, /*  a10 */ 
   0xf400, 0x4057, 0x3702, 0x0024, 0x2000, 0x0000, 0x0000, 0x0024, /*  a18 */ 
   0x0006, 0x0c10, 0x6890, 0x0024, 0x2801, 0xf6c0, 0x3800, 0x0024, /*  a20 */ 
   0x0000, 0x3000, 0x2801, 0x7e40, 0x0006, 0x0d10, 0x0000, 0x9000, /*  a28 */ 
   0x0006, 0x0d10, 0x3453, 0x0024, 0x34b0, 0x4024, 0x4192, 0x0024, /*  a30 */ 
   0x4192, 0x0024, 0x2801, 0x7e40, 0xa010, 0x0024, 0x0000, 0x0201, /*  a38 */ 
   0x0006, 0x0d10, 0x2900, 0xb680, 0x3613, 0x0024, 0x0006, 0x8081, /*  a40 */ 
   0x3473, 0x0024, 0x3423, 0x0024, 0x3c50, 0x0024, 0x34b0, 0x8024, /*  a48 */ 
   0x3450, 0xc024, 0xd234, 0x0024, 0x0000, 0x3fc3, 0xb234, 0x0024, /*  a50 */ 
   0x4122, 0x0024, 0xf400, 0x4055, 0x3500, 0x0024, 0x3cb0, 0x0024, /*  a58 */ 
   0x3490, 0x0024, 0x2801, 0x7e40, 0x6090, 0x138c, 0x003f, 0xfe04, /*  a60 */ 
   0x0000, 0x0401, 0x0006, 0x0d10, 0x2900, 0xb680, 0x3613, 0x0024, /*  a68 */ 
   0x0006, 0x8081, 0x3473, 0x0024, 0x3423, 0x0024, 0x3c50, 0x0024, /*  a70 */ 
   0x34b0, 0x8024, 0x3400, 0xc024, 0xa346, 0x0024, 0xd234, 0x0024, /*  a78 */ 
   0x0000, 0x3fc3, 0xb234, 0x0024, 0x4122, 0x1142, 0xf400, 0x4055, /*  a80 */ 
   0x0006, 0x8081, 0x3500, 0x0024, 0xd024, 0x3000, 0xb234, 0x0024, /*  a88 */ 
   0x4122, 0x0024, 0xf400, 0x4055, 0x3500, 0x0024, 0x3cb0, 0x0024, /*  a90 */ 
   0x3490, 0x0024, 0x2801, 0x7e40, 0x6090, 0x138c, 0x0000, 0x4000, /*  a98 */ 
   0x0000, 0x0202, 0x0006, 0x0d10, 0x3453, 0x0024, 0x34b0, 0x4024, /*  aa0 */ 
   0x6122, 0x0024, 0xa010, 0x0024, 0x0004, 0x8001, 0x3800, 0x118c, /*  aa8 */ 
   0x0006, 0x0d10, 0x3000, 0x0024, 0x6012, 0x1280, 0x0000, 0x0401, /*  ab0 */ 
   0x2801, 0x8109, 0x0000, 0x0024, 0x6890, 0x82cc, 0x2801, 0xf6c0, /*  ab8 */ 
   0x3800, 0x0024, 0x6012, 0x0024, 0x0006, 0x07c1, 0x2801, 0xa941, /*  ac0 */ 
   0x4012, 0x0024, 0xf400, 0x4057, 0x3702, 0x0024, 0x2000, 0x0000, /*  ac8 */ 
   0x0000, 0x0024, 0x2801, 0xa940, 0x0000, 0x0024, 0x0016, 0x2200, /*  ad0 */ 
   0x0006, 0x0d50, 0x6892, 0x2040, 0x2801, 0xa940, 0x38f0, 0x4024, /*  ad8 */ 
   0x002c, 0x4400, 0x0000, 0x0081, 0x0006, 0x0d50, 0x3810, 0x0024, /*  ae0 */ 
   0x2801, 0xa940, 0x38f0, 0x4024, 0x003b, 0x8000, 0x0000, 0x0081, /*  ae8 */ 
   0x0006, 0x0d50, 0x3810, 0x0024, 0x2801, 0xa940, 0x38f0, 0x4024, /*  af0 */ 
   0x0007, 0xd000, 0x0006, 0x0d50, 0xb882, 0x2040, 0x2801, 0xa940, /*  af8 */ 
   0x38f0, 0x4024, 0x000f, 0xa000, 0x0006, 0x0d50, 0xb882, 0x2040, /*  b00 */ 
   0x2801, 0xa940, 0x38f0, 0x4024, 0x0015, 0x8880, 0x0006, 0x0d50, /*  b08 */ 
   0xb882, 0x2040, 0x2801, 0xa940, 0x38f0, 0x4024, 0x0017, 0x7000, /*  b10 */ 
   0x0006, 0x0d50, 0xb882, 0x2040, 0x2801, 0xa940, 0x38f0, 0x4024, /*  b18 */ 
   0x001f, 0x4000, 0x0006, 0x0d50, 0xb882, 0x2040, 0x2801, 0xa940, /*  b20 */ 
   0x38f0, 0x4024, 0x002b, 0x1100, 0x0006, 0x0d50, 0xb882, 0x2040, /*  b28 */ 
   0x2801, 0xa940, 0x38f0, 0x4024, 0x002e, 0xe000, 0x0006, 0x0d50, /*  b30 */ 
   0xb882, 0x2040, 0x2801, 0xa940, 0x38f0, 0x4024, 0x001d, 0xc000, /*  b38 */ 
   0x0006, 0x0d50, 0x6892, 0x2040, 0x2801, 0xa940, 0x38f0, 0x4024, /*  b40 */ 
   0x0006, 0x0d50, 0x0000, 0x0201, 0x0000, 0xfa04, 0x2900, 0xb680, /*  b48 */ 
   0x3613, 0x0024, 0x0006, 0x8081, 0xb88a, 0x11cc, 0x3423, 0x0024, /*  b50 */ 
   0x3c50, 0x0024, 0x34b0, 0x8024, 0x3450, 0xc024, 0xd234, 0x0024, /*  b58 */ 
   0x0000, 0x3fc3, 0xb234, 0x0024, 0x4122, 0x0024, 0xf400, 0x4055, /*  b60 */ 
   0x3500, 0x0024, 0x3cb0, 0x0024, 0x3490, 0x0024, 0xfe50, 0x4005, /*  b68 */ 
   0x48b2, 0x938c, 0xfeca, 0x0024, 0x40b2, 0x0024, 0x3810, 0x0024, /*  b70 */ 
   0x2801, 0xa940, 0x38f0, 0x4024, 0x003f, 0xfe04, 0x0000, 0x0401, /*  b78 */ 
   0x0006, 0x0d50, 0x2900, 0xb680, 0x3613, 0x0024, 0x0006, 0x8081, /*  b80 */ 
   0x3473, 0x0024, 0x3423, 0x0024, 0x3c50, 0x0024, 0x34b0, 0x8024, /*  b88 */ 
   0x3400, 0xc024, 0xa346, 0x0024, 0xd234, 0x0024, 0x0000, 0x3fc3, /*  b90 */ 
   0xb234, 0x0024, 0x4122, 0x1142, 0xf400, 0x4055, 0x0006, 0x8081, /*  b98 */ 
   0x3500, 0x0024, 0xd024, 0x3000, 0xb234, 0x0024, 0x4122, 0x0024, /*  ba0 */ 
   0xf400, 0x4055, 0x0000, 0x0041, 0x3500, 0x0024, 0x3cb0, 0x0024, /*  ba8 */ 
   0x3490, 0x0024, 0xfe02, 0x938c, 0x48b2, 0x0024, 0x3810, 0x0024, /*  bb0 */ 
   0x2801, 0xa940, 0x38f0, 0x4024, 0x003f, 0xfe04, 0x0000, 0x0401, /*  bb8 */ 
   0x0006, 0x0d50, 0x2900, 0xb680, 0x3613, 0x0024, 0x0006, 0x8081, /*  bc0 */ 
   0x3473, 0x0024, 0x3423, 0x0024, 0x3c50, 0x0024, 0x34b0, 0x8024, /*  bc8 */ 
   0x3400, 0xc024, 0xa346, 0x0024, 0xd234, 0x0024, 0x0000, 0x3fc3, /*  bd0 */ 
   0xb234, 0x0024, 0x4122, 0x1142, 0xf400, 0x4055, 0x0006, 0x8081, /*  bd8 */ 
   0x3500, 0x0024, 0xd024, 0x3000, 0xb234, 0x0024, 0x4122, 0x0024, /*  be0 */ 
   0xf400, 0x4055, 0x3500, 0x0024, 0x3cb0, 0x0024, 0x0000, 0x0280, /*  be8 */ 
   0x3490, 0x4024, 0xfe02, 0x938c, 0x48b2, 0x0024, 0x3810, 0x0024, /*  bf0 */ 
   0x2801, 0xa940, 0x38f0, 0x4024, 0x0006, 0x0bd0, 0x6890, 0x0024, /*  bf8 */ 
   0x2801, 0xf6c0, 0x3800, 0x0024, 0x0000, 0x0201, 0x2900, 0xb680, /*  c00 */ 
   0x3613, 0x11cc, 0x3423, 0x0024, 0x3c50, 0x0024, 0x3490, 0x4024, /*  c08 */ 
   0x6014, 0x124c, 0x0000, 0x0081, 0x2801, 0xacc5, 0x0006, 0x0c90, /*  c10 */ 
   0x0006, 0x0bd0, 0x6890, 0x0024, 0x2801, 0xf6c0, 0x3800, 0x0024, /*  c18 */ 
   0x3000, 0x0024, 0x6012, 0x0024, 0x0000, 0x0241, 0x2801, 0xaec9, /*  c20 */ 
   0x0000, 0x0024, 0x6890, 0x034c, 0x2801, 0xf6c0, 0x3800, 0x0024, /*  c28 */ 
   0x0006, 0x0cd0, 0x30e1, 0x184c, 0x3000, 0x0024, 0x6012, 0x0024, /*  c30 */ 
   0x0008, 0x0001, 0x2801, 0xb0d5, 0x0000, 0x0024, 0x6498, 0x0024, /*  c38 */ 
   0x3e10, 0x4024, 0x0000, 0x0081, 0x2900, 0x5580, 0x3e01, 0x0024, /*  c40 */ 
   0x36e3, 0x004c, 0x3000, 0x0024, 0x6012, 0x0024, 0x000b, 0x8011, /*  c48 */ 
   0x2801, 0xbc95, 0x0006, 0x0cd2, 0x0000, 0x0201, 0x0004, 0x0010, /*  c50 */ 
   0x2915, 0x8300, 0x0001, 0x0000, 0x000b, 0x8011, 0x0005, 0x0010, /*  c58 */ 
   0x291f, 0xc6c0, 0x0001, 0x0000, 0x0006, 0x0cd0, 0x30e1, 0x0024, /*  c60 */ 
   0x3000, 0x0024, 0x6012, 0x0024, 0x0000, 0x0281, 0x2801, 0xb805, /*  c68 */ 
   0x6012, 0x0024, 0x000b, 0x8001, 0x2801, 0xb895, 0x3613, 0x0024, /*  c70 */ 
   0x36f3, 0x0024, 0x000b, 0x8001, 0x6498, 0x184c, 0x0006, 0x0cd2, /*  c78 */ 
   0x0003, 0x8000, 0x3e10, 0x4024, 0x2900, 0x5580, 0x3e01, 0x0024, /*  c80 */ 
   0x36f3, 0x0024, 0x3009, 0x3844, 0x3e10, 0x0024, 0x0000, 0x0400, /*  c88 */ 
   0x3000, 0x8024, 0x0008, 0x0010, 0x3e00, 0x8024, 0x3201, 0x0024, /*  c90 */ 
   0x6408, 0x4051, 0x2902, 0x2700, 0x0001, 0xbf48, 0x0000, 0x0400, /*  c98 */ 
   0x0000, 0x0011, 0x3613, 0x008c, 0x30d0, 0x7844, 0x3e10, 0x4024, /*  ca0 */ 
   0x3000, 0x8024, 0x0008, 0x0010, 0x3e00, 0x8024, 0x3201, 0x0024, /*  ca8 */ 
   0x2902, 0x2700, 0x6408, 0x0024, 0x0006, 0x8a10, 0x0000, 0x01c1, /*  cb0 */ 
   0x36e3, 0x0000, 0xb010, 0x9bc4, 0x0000, 0x0024, 0x2801, 0xc305, /*  cb8 */ 
   0x6192, 0x0024, 0x2900, 0xb680, 0x6102, 0x184c, 0x4088, 0x0024, /*  cc0 */ 
   0x0000, 0x0024, 0x2801, 0xc305, 0x0000, 0x0024, 0x6890, 0x0b4c, /*  cc8 */ 
   0x3a00, 0x0024, 0x3009, 0x0000, 0x4080, 0x0024, 0x0000, 0x0024, /*  cd0 */ 
   0x2902, 0x5b45, 0x0001, 0xc808, 0x0006, 0x9f92, 0x0000, 0x4003, /*  cd8 */ 
   0x3009, 0x0811, 0x3100, 0x8024, 0xffa6, 0x0024, 0x48b6, 0x0024, /*  ce0 */ 
   0x2902, 0x5b40, 0x4384, 0x0024, 0x2902, 0x52c0, 0x3613, 0x0024, /*  ce8 */ 
   0x4082, 0x184c, 0x2902, 0x57c0, 0x3e00, 0x0024, 0x2902, 0x5b40, /*  cf0 */ 
   0x36f3, 0x0024, 0x0000, 0x0401, 0x3473, 0x184c, 0x3423, 0x0024, /*  cf8 */ 
   0x2900, 0xb680, 0x3c50, 0x0024, 0x3c90, 0x0024, 0x290b, 0x1400, /*  d00 */ 
   0x3493, 0x0024, 0x4080, 0x0024, 0x0000, 0x0024, 0x2801, 0xf355, /*  d08 */ 
   0x0000, 0x0024, 0x3473, 0x0024, 0x3423, 0x0024, 0x3450, 0x0024, /*  d10 */ 
   0x3490, 0x4024, 0x6014, 0x134c, 0x0006, 0x0bd1, 0x2801, 0xcdc5, /*  d18 */ 
   0x34c0, 0x0024, 0x6890, 0x0024, 0x2801, 0xf6c0, 0x3900, 0x0024, /*  d20 */ 
   0x4080, 0x4512, 0x0006, 0x0d11, 0x2801, 0xd245, 0x0000, 0xfa02, /*  d28 */ 
   0xf400, 0x4511, 0x0000, 0xfa02, 0x0006, 0x0d52, 0x3210, 0x05cc, /*  d30 */ 
   0x32f0, 0x44cc, 0x3111, 0x8024, 0x31f1, 0xc024, 0xfef4, 0x4087, /*  d38 */ 
   0x48b6, 0x0024, 0xfeee, 0x0024, 0x2914, 0xa580, 0x42b6, 0x0024, /*  d40 */ 
   0x2801, 0xd5c0, 0x0007, 0x89d0, 0x3111, 0x89cc, 0x3110, 0x08cc, /*  d48 */ 
   0x3211, 0x0024, 0x32f1, 0x4024, 0xfedc, 0x4185, 0x48be, 0x07c1, /*  d50 */ 
   0xfeca, 0x0024, 0x46be, 0x0024, 0xfef4, 0x4087, 0x48b6, 0x0024, /*  d58 */ 
   0xfeee, 0x0024, 0x2914, 0xa580, 0x42b6, 0x0024, 0x0007, 0x89d0, /*  d60 */ 
   0x0006, 0x0d51, 0x3810, 0x184c, 0x38f0, 0x4024, 0x3110, 0x8024, /*  d68 */ 
   0x428a, 0x0703, 0x2908, 0xbe80, 0x3101, 0x0024, 0xf400, 0x4512, /*  d70 */ 
   0x0000, 0x03d3, 0x3123, 0x0a0c, 0x3100, 0x0024, 0x000b, 0x8011, /*  d78 */ 
   0x3a00, 0x0024, 0x2801, 0xf140, 0x0008, 0x0012, 0x0000, 0x03d5, /*  d80 */ 
   0x3483, 0x0024, 0x1fff, 0xfc55, 0x3480, 0x0024, 0x6402, 0x0024, /*  d88 */ 
   0x0006, 0x0c90, 0x2801, 0xdbc9, 0x0000, 0x0024, 0xf400, 0x4004, /*  d90 */ 
   0x3000, 0x0024, 0x4090, 0x0024, 0xf400, 0x4480, 0x2801, 0xe195, /*  d98 */ 
   0x000b, 0x8001, 0x0000, 0x0005, 0x6540, 0x0024, 0x0000, 0x0024, /*  da0 */ 
   0x2801, 0xecd8, 0x4490, 0x0024, 0x2401, 0xe0c0, 0x0000, 0x0024, /*  da8 */ 
   0x0006, 0x0dc1, 0x4554, 0x0800, 0x4122, 0x0024, 0x659a, 0x4055, /*  db0 */ 
   0x0006, 0x0e01, 0x4122, 0x3400, 0xf400, 0x4055, 0x3210, 0x0024, /*  db8 */ 
   0x3d00, 0x0024, 0x2801, 0xecc0, 0x0000, 0x0024, 0x6014, 0x0024, /*  dc0 */ 
   0x0001, 0x0000, 0x2801, 0xe915, 0x0000, 0x0005, 0x0008, 0x0012, /*  dc8 */ 
   0x0008, 0x0010, 0x0003, 0x8001, 0x0006, 0x0d13, 0x3613, 0x0024, /*  dd0 */ 
   0x3009, 0x3811, 0x2902, 0x9740, 0x0004, 0x0011, 0x0008, 0x0010, /*  dd8 */ 
   0x0001, 0x0000, 0x291f, 0xc6c0, 0x0005, 0x0011, 0x000f, 0x0011, /*  de0 */ 
   0x0008, 0x0010, 0x33d0, 0x184c, 0x6010, 0xb844, 0x3e10, 0x0024, /*  de8 */ 
   0x0000, 0x0400, 0x3320, 0x4024, 0x3e00, 0x4024, 0x3301, 0x0024, /*  df0 */ 
   0x2902, 0x2700, 0x6408, 0x0024, 0x36e3, 0x0024, 0x3009, 0x1bc4, /*  df8 */ 
   0x3009, 0x1bd1, 0x6540, 0x0024, 0x0000, 0x0024, 0x2801, 0xecd8, /*  e00 */ 
   0x4490, 0x0024, 0x2401, 0xec80, 0x0000, 0x0024, 0x0006, 0x0dc1, /*  e08 */ 
   0x4554, 0x0840, 0x4122, 0x0024, 0x659a, 0x4055, 0x0006, 0x0e01, /*  e10 */ 
   0x4122, 0x3400, 0xf400, 0x4055, 0x3110, 0x0024, 0x3d00, 0x0024, /*  e18 */ 
   0xf400, 0x4510, 0x0030, 0x0013, 0x3073, 0x184c, 0x3073, 0x0024, /*  e20 */ 
   0x3013, 0x0024, 0x3000, 0x4024, 0x6140, 0x0024, 0x0000, 0x0201, /*  e28 */ 
   0x3800, 0x0024, 0x0006, 0x0dc0, 0x3e11, 0x0024, 0x290c, 0x7300, /*  e30 */ 
   0x3e10, 0x0024, 0x3300, 0x1b8c, 0xb010, 0x0024, 0x0000, 0x0024, /*  e38 */ 
   0x2801, 0xf355, 0x0000, 0x0024, 0x0000, 0x03d5, 0x3483, 0x0024, /*  e40 */ 
   0x1fff, 0xfc55, 0x3480, 0x0024, 0x4080, 0x0024, 0x0000, 0x0804, /*  e48 */ 
   0x2801, 0xd995, 0x0000, 0x0024, 0x0006, 0x8012, 0x0006, 0x8010, /*  e50 */ 
   0x0006, 0x0d51, 0x3010, 0x984c, 0x30f0, 0xc024, 0x0000, 0x0021, /*  e58 */ 
   0xf2d6, 0x07c6, 0x290a, 0xf5c0, 0x4682, 0x0400, 0x6894, 0x0840, /*  e60 */ 
   0xb886, 0x0bc1, 0xbcd6, 0x0024, 0x3a10, 0x8024, 0x3af0, 0xc024, /*  e68 */ 
   0x36f3, 0x4024, 0x36f3, 0xd80e, 0x36f4, 0x9813, 0x36f4, 0x1811, /*  e70 */ 
   0x36f1, 0x9807, 0x36f1, 0x1805, 0x36f0, 0x9803, 0x36f0, 0x1801, /*  e78 */ 
   0x3405, 0x9014, 0x36f3, 0x0024, 0x36f2, 0x1815, 0x2000, 0x0000, /*  e80 */ 
   0x36f2, 0x9817, 0x0007, 0x0001, 0x8271, 0x0006, 0x00d2, 0x3613, /*  e88 */ 
   0x0024, 0x3e12, 0xb817, 0x3e12, 0x3815, 0x3e05, 0xb814, 0x3615, /*  e90 */ 
   0x0024, 0x0000, 0x800a, 0x3e10, 0x3801, 0x0019, 0x9300, 0x3e14, /*  e98 */ 
   0x3811, 0x0030, 0x0250, 0x3e04, 0xb813, 0x2900, 0xbdc0, 0x3800, /*  ea0 */ 
   0x0024, 0x2902, 0x6cc0, 0x0000, 0x0200, 0xb882, 0x0024, 0x2914, /*  ea8 */ 
   0xbec0, 0x0006, 0x0bd0, 0x0000, 0x1540, 0x0007, 0x8190, 0x2900, /*  eb0 */ 
   0x1c80, 0x3800, 0x0024, 0x4080, 0x0024, 0x0006, 0x0bd1, 0x2800, /*  eb8 */ 
   0xa815, 0x0000, 0x0024, 0x3100, 0x0024, 0x4080, 0x0024, 0x0030, /*  ec0 */ 
   0x0010, 0x2800, 0xa815, 0x0000, 0x0201, 0x3000, 0x0024, 0xb010, /*  ec8 */ 
   0x0024, 0x0000, 0x0024, 0x2800, 0xa815, 0x0000, 0x0024, 0x2900, /*  ed0 */ 
   0x1c80, 0x0000, 0x0024, 0x4080, 0x0024, 0x0006, 0x0bd0, 0x2800, /*  ed8 */ 
   0xa815, 0x0000, 0x0024, 0x3000, 0x0024, 0x4080, 0x0024, 0x0000, /*  ee0 */ 
   0x0201, 0x2800, 0xa405, 0x0030, 0x0010, 0x0030, 0x0010, 0x0000, /*  ee8 */ 
   0x0201, 0x3000, 0x0024, 0xb010, 0x0024, 0x0000, 0x0024, 0x2902, /*  ef0 */ 
   0x5255, 0x0000, 0xb0c8, 0x0006, 0x0bd1, 0x3100, 0x0024, 0x4080, /*  ef8 */ 
   0x0024, 0x0000, 0x0024, 0x2800, 0xaf05, 0x0000, 0x0024, 0x0007, /*  f00 */ 
   0x8a52, 0x3200, 0x0024, 0x4080, 0x0024, 0x0000, 0x0024, 0x2800, /*  f08 */ 
   0xaf09, 0x0000, 0x0024, 0xf292, 0x0800, 0x6012, 0x0024, 0x0000, /*  f10 */ 
   0x0000, 0x2800, 0xaec5, 0x0000, 0x0024, 0x3200, 0x0024, 0x4090, /*  f18 */ 
   0x0024, 0xb880, 0x2800, 0x3900, 0x0024, 0x3100, 0x0024, 0x4080, /*  f20 */ 
   0x0024, 0x0000, 0x0024, 0x2901, 0x2e05, 0x0000, 0xa808, 0x2902, /*  f28 */ 
   0x5240, 0x0000, 0x0024, 0x0000, 0x0010, 0x0006, 0x9f51, 0x0006, /*  f30 */ 
   0x9f92, 0x0030, 0x0493, 0x0000, 0x0201, 0x6890, 0xa410, 0x3b00, /*  f38 */ 
   0x2810, 0x0006, 0x8a10, 0x3009, 0x0000, 0x6012, 0x0024, 0x0006, /*  f40 */ 
   0x9fd0, 0x2800, 0xb448, 0xb880, 0x0024, 0x6890, 0x0024, 0x3009, /*  f48 */ 
   0x2000, 0x36f4, 0x9813, 0x36f4, 0x1811, 0x36f0, 0x1801, 0x3405, /*  f50 */ 
   0x9014, 0x36f3, 0x0024, 0x36f2, 0x1815, 0x2000, 0x0000, 0x36f2, /*  f58 */ 
   0x9817, 0x0007, 0x0001, 0x5840, 0x0006, 0x0001, 0x0001, 0x0006, /*  f60 */ 
   0x8007, 0x0000, 0x0006, 0x0018, 0x0002, 0x0000, 0xffff, 0xffff, /*  f68 */ 
   0x0000, 0x0000, 0x0000, 0x0000, 0x0003, 0x0000, 0xfffd, 0xffff, /*  f70 */ 
   0x0001, 0x0000, 0x0000, 0x0000, 0x0004, 0x0000, 0xfffa, 0xffff, /*  f78 */ 
   0x0004, 0x0000, 0xffff, 0xffff, 0x0007, 0x0001, 0x87e8, 0x0006, /*  f80 */ 
   0x004e, 0x3613, 0x0024, 0x3e12, 0xb815, 0x0000, 0x800a, 0x3e14, /*  f88 */ 
   0x7813, 0x3e10, 0xb803, 0x3e11, 0x3805, 0x3e11, 0xb807, 0x3e13, /*  f90 */ 
   0xf80e, 0x6812, 0x0024, 0x3e03, 0x7810, 0x0fff, 0xffd3, 0x0000, /*  f98 */ 
   0x0091, 0xbd86, 0x9850, 0x3e10, 0x3804, 0x3e00, 0x7812, 0xbe8a, /*  fa0 */ 
   0x8bcc, 0x409e, 0x8086, 0x2401, 0xff07, 0xfe49, 0x2821, 0x526a, /*  fa8 */ 
   0x8801, 0x5c87, 0x280e, 0x4eba, 0x9812, 0x4286, 0x40e1, 0xb284, /*  fb0 */ 
   0x1bc1, 0x4de6, 0x0024, 0xad17, 0x2627, 0x4fde, 0x9804, 0x4498, /*  fb8 */ 
   0x1bc0, 0x0000, 0x0024, 0x2801, 0xfd15, 0x3a11, 0xa807, 0x36f3, /*  fc0 */ 
   0x4024, 0x36f3, 0xd80e, 0x36f1, 0x9807, 0x36f1, 0x1805, 0x36f0, /*  fc8 */ 
   0x9803, 0x36f4, 0x5813, 0x2000, 0x0000, 0x36f2, 0x9815, 0x0007, /*  fd0 */ 
   0x0001, 0x880f, 0x0006, 0x0056, 0x3613, 0x0024, 0x3e12, 0xb815, /*  fd8 */ 
   0x0000, 0x800a, 0x3e10, 0xb803, 0x3e11, 0x3805, 0x3e11, 0xb807, /*  fe0 */ 
   0x3e13, 0xf80e, 0x6812, 0x0024, 0x3e03, 0x7810, 0x3009, 0x1850, /*  fe8 */ 
   0x3e10, 0x3804, 0x3e10, 0x7812, 0x32f3, 0x0024, 0xbd86, 0x0024, /*  ff0 */ 
   0x4091, 0xe2e3, 0x3009, 0x0046, 0x2402, 0x0a80, 0x3009, 0x0047, /*  ff8 */ 
   0x32f0, 0x0801, 0xfe1f, 0x6465, 0x5e8a, 0x0024, 0x44ba, 0x0024, /* 1000 */ 
   0xfee2, 0x0024, 0x5d8a, 0x1800, 0x4482, 0x4160, 0x48ba, 0x8046, /* 1008 */ 
   0x4dc6, 0x1822, 0x4de6, 0x8047, 0x36f3, 0x0024, 0x36f0, 0x5812, /* 1010 */ 
   0xad17, 0x2627, 0x4fde, 0x9804, 0x4498, 0x1bc0, 0x0000, 0x0024, /* 1018 */ 
   0x2802, 0x0615, 0x3a11, 0xa807, 0x36f3, 0x4024, 0x36f3, 0xd80e, /* 1020 */ 
   0x36f1, 0x9807, 0x36f1, 0x1805, 0x36f0, 0x9803, 0x2000, 0x0000, /* 1028 */ 
   0x36f2, 0x9815, 0x0007, 0x0001, 0x883a, 0x0006, 0x004e, 0x3613, /* 1030 */ 
   0x0024, 0x3e10, 0xb810, 0x3e11, 0x3805, 0x3e02, 0x0024, 0x0030, /* 1038 */ 
   0x0010, 0xce9a, 0x0002, 0x0000, 0x0200, 0x2902, 0x1840, 0xb024, /* 1040 */ 
   0x0024, 0xc020, 0x0024, 0x0000, 0x0200, 0x2802, 0x1045, 0x6e9a, /* 1048 */ 
   0x0002, 0x4182, 0x0024, 0x0000, 0x0400, 0x2802, 0x1685, 0xae1a, /* 1050 */ 
   0x0024, 0x6104, 0x984c, 0x0000, 0x0024, 0x2900, 0xb689, 0x0002, /* 1058 */ 
   0x1648, 0x3613, 0x0024, 0x6103, 0xe4e5, 0x2900, 0xb680, 0x408a, /* 1060 */ 
   0x184c, 0x2900, 0xb680, 0x408a, 0x4141, 0x3601, 0x0024, 0x4582, /* 1068 */ 
   0x9bc5, 0x2802, 0x1680, 0xceca, 0x0024, 0xc408, 0x0024, 0xf2e2, /* 1070 */ 
   0x1bc8, 0x36f1, 0x1805, 0x2000, 0x0011, 0x36f0, 0x9810, 0x68c2, /* 1078 */ 
   0x0024, 0x2000, 0x0000, 0x4c92, 0x0024, 0x0007, 0x0001, 0x8861, /* 1080 */ 
   0x0006, 0x0076, 0x0006, 0x8a17, 0x3613, 0x1c00, 0x6093, 0xe1e3, /* 1088 */ 
   0x0000, 0x03c3, 0x0006, 0x9f95, 0xb132, 0x9415, 0x3500, 0xfc01, /* 1090 */ 
   0x2802, 0x2655, 0xa306, 0x0024, 0x0007, 0x0097, 0x003f, 0xc001, /* 1098 */ 
   0x3500, 0x184c, 0xb011, 0xe4e5, 0xb182, 0x1c04, 0xd400, 0x184c, /* 10a0 */ 
   0x0000, 0x0205, 0xac52, 0x3802, 0x0006, 0xc082, 0x4212, 0x0024, /* 10a8 */ 
   0xf400, 0x4057, 0xb182, 0x1c04, 0xd400, 0x0024, 0xac52, 0x1404, /* 10b0 */ 
   0xd142, 0x0024, 0x0000, 0x3fc4, 0xb142, 0x0024, 0x4122, 0x1bc2, /* 10b8 */ 
   0xf400, 0x4057, 0x3700, 0x4024, 0xd101, 0x6465, 0x0007, 0x0097, /* 10c0 */ 
   0x3f00, 0x3814, 0x0025, 0xffd4, 0x0006, 0x8017, 0x3710, 0x160c, /* 10c8 */ 
   0x0006, 0x9f94, 0x37f0, 0x73d5, 0x6c92, 0x3808, 0x3f10, 0x0024, /* 10d0 */ 
   0x3ff0, 0x4024, 0x3009, 0x1040, 0x3009, 0x13c1, 0x6010, 0x0024, /* 10d8 */ 
   0x0000, 0x0024, 0x2902, 0x7645, 0x0002, 0x2248, 0x2802, 0x2494, /* 10e0 */ 
   0x0006, 0x0001, 0x4010, 0x0024, 0x0005, 0xf601, 0x6010, 0x0024, /* 10e8 */ 
   0x0000, 0x0040, 0x2802, 0x2614, 0x0030, 0x0497, 0x3f00, 0x0024, /* 10f0 */ 
   0x36f2, 0x1814, 0x4330, 0x9803, 0x2000, 0x0000, 0x8880, 0x1bc1, /* 10f8 */ 
   0x0007, 0x0001, 0x889c, 0x0006, 0x00a4, 0x3613, 0x0024, 0x3e22, /* 1100 */ 
   0xb806, 0x3e05, 0xb814, 0x3615, 0x0024, 0x0000, 0x800a, 0x3e10, /* 1108 */ 
   0x3801, 0x3e10, 0xb803, 0x3e11, 0x7807, 0x6848, 0x930c, 0x3411, /* 1110 */ 
   0x780d, 0x459a, 0x10c0, 0x0000, 0x0201, 0x6012, 0x384e, 0x0000, /* 1118 */ 
   0x0241, 0x2802, 0x2d95, 0x6012, 0xb80f, 0x2402, 0x2cc5, 0x0000, /* 1120 */ 
   0x0024, 0x3000, 0x0001, 0x3101, 0x8407, 0x6cfe, 0x0024, 0xac42, /* 1128 */ 
   0x0024, 0xaf4e, 0x2040, 0x3911, 0x8024, 0x2802, 0x3940, 0x0000, /* 1130 */ 
   0x0024, 0x0000, 0x0281, 0x2802, 0x30d5, 0x6012, 0x4455, 0x2402, /* 1138 */ 
   0x3005, 0x0000, 0x0024, 0x3000, 0x0001, 0x3101, 0x8407, 0x4cf2, /* 1140 */ 
   0x0024, 0xac42, 0x0024, 0xaf4e, 0x2040, 0x3911, 0x8024, 0x2802, /* 1148 */ 
   0x3940, 0x0000, 0x0024, 0x0000, 0x0024, 0x2802, 0x3515, 0x4080, /* 1150 */ 
   0x0024, 0x3110, 0x0401, 0xf20f, 0x0203, 0x2402, 0x3445, 0x8dd6, /* 1158 */ 
   0x0024, 0x4dce, 0x0024, 0xf1fe, 0x0024, 0xaf4e, 0x0024, 0x6dc6, /* 1160 */ 
   0x2046, 0xf1df, 0x0203, 0xaf4f, 0x1011, 0xf20e, 0x07cc, 0x8dd6, /* 1168 */ 
   0x2486, 0x2802, 0x3940, 0x0000, 0x0024, 0x0000, 0x0024, 0x2802, /* 1170 */ 
   0x3795, 0x0000, 0x0024, 0x0fff, 0xffd1, 0x2402, 0x36c5, 0x3010, /* 1178 */ 
   0x0001, 0xac4f, 0x0801, 0x3821, 0x8024, 0x2802, 0x3940, 0x0000, /* 1180 */ 
   0x0024, 0x0fff, 0xffd1, 0x2402, 0x3905, 0x3010, 0x0001, 0x3501, /* 1188 */ 
   0x9407, 0xac47, 0x0801, 0xaf4e, 0x2082, 0x3d11, 0x8024, 0x36f3, /* 1190 */ 
   0xc024, 0x36f3, 0x980d, 0x36f1, 0x5807, 0x36f0, 0x9803, 0x36f0, /* 1198 */ 
   0x1801, 0x3405, 0x9014, 0x36e3, 0x0024, 0x2000, 0x0000, 0x36f2, /* 11a0 */ 
   0x9806, 0x0007, 0x0001, 0x88ee, 0x0006, 0x005c, 0x0006, 0x9f97, /* 11a8 */ 
   0x3e00, 0x5c15, 0x0007, 0x0097, 0x003f, 0xc001, 0x3500, 0x3840, /* 11b0 */ 
   0xb011, 0xe4e5, 0xb182, 0x1c04, 0xd400, 0x184c, 0x0000, 0x0205, /* 11b8 */ 
   0xac52, 0x3802, 0x0006, 0xc082, 0x4212, 0x0024, 0xb182, 0x4057, /* 11c0 */ 
   0x3701, 0x0024, 0xd400, 0x0024, 0xac52, 0x1404, 0xd142, 0x0024, /* 11c8 */ 
   0x0000, 0x3fc4, 0xb142, 0x0024, 0x4122, 0x1bc2, 0xf400, 0x4057, /* 11d0 */ 
   0x3700, 0x4024, 0xd101, 0x6465, 0x0007, 0x0097, 0x3f00, 0x3814, /* 11d8 */ 
   0x0025, 0xffd4, 0x0006, 0x8017, 0x3710, 0x160c, 0x0006, 0x9f94, /* 11e0 */ 
   0x37f0, 0x73d5, 0x6c92, 0x0024, 0x3f10, 0x1040, 0x3ff0, 0x53c1, /* 11e8 */ 
   0x6010, 0x0024, 0x0000, 0x0024, 0x2802, 0x4514, 0x0006, 0x0001, /* 11f0 */ 
   0x4010, 0x0024, 0x0005, 0xf601, 0x6010, 0x9bd4, 0x0000, 0x0040, /* 11f8 */ 
   0x2802, 0x4694, 0x0030, 0x0497, 0x3f00, 0x0024, 0x2000, 0x0000, /* 1200 */ 
   0x36f0, 0x5800, 0x0007, 0x0001, 0x82da, 0x0006, 0x003a, 0x3e10, /* 1208 */ 
   0xb812, 0x3e11, 0xb810, 0x3e12, 0x0024, 0x0006, 0x9f92, 0x0025, /* 1210 */ 
   0xffd0, 0x3e04, 0x4bd1, 0x3181, 0xf847, 0xb68c, 0x4440, 0x3009, /* 1218 */ 
   0x0802, 0x6024, 0x3806, 0x0006, 0x8a10, 0x2902, 0x7645, 0x0000, /* 1220 */ 
   0xb888, 0x0000, 0x0800, 0x6101, 0x1602, 0xaf2e, 0x0024, 0x4214, /* 1228 */ 
   0x1be3, 0xaf0e, 0x1811, 0x0fff, 0xfc00, 0xb200, 0x9bc7, 0x0000, /* 1230 */ 
   0x03c0, 0x2800, 0xbcc5, 0xb204, 0xa002, 0x2902, 0x3b80, 0x3613, /* 1238 */ 
   0x2002, 0x4680, 0x1bc8, 0x36f1, 0x9810, 0x2000, 0x0000, 0x36f0, /* 1240 */ 
   0x9812, 0x0007, 0x0001, 0x891c, 0x0006, 0x0022, 0x0000, 0x0400, /* 1248 */ 
   0x6102, 0x0024, 0x3e11, 0x3805, 0x2802, 0x4a09, 0x3e02, 0x0024, /* 1250 */ 
   0x2900, 0xb680, 0x408a, 0x188c, 0x2900, 0xb680, 0x408a, 0x4141, /* 1258 */ 
   0x4582, 0x1bc8, 0x2000, 0x0000, 0x36f1, 0x1805, 0x2900, 0xb680, /* 1260 */ 
   0x4102, 0x184c, 0xb182, 0x1bc8, 0x2000, 0x0000, 0x36f1, 0x1805, /* 1268 */ 
   0x0007, 0x0001, 0x892d, 0x0006, 0x0036, 0x2a02, 0x4b8e, 0x3e12, /* 1270 */ 
   0xb817, 0x3e10, 0x3802, 0x0006, 0x9f97, 0x3009, 0x1fc2, 0x3e04, /* 1278 */ 
   0x5c00, 0x6020, 0xb810, 0x0030, 0x0451, 0x2802, 0x4e14, 0x0006, /* 1280 */ 
   0x0002, 0x4020, 0x0024, 0x0005, 0xfb02, 0x6024, 0x0024, 0x0025, /* 1288 */ 
   0xffd0, 0x2802, 0x5051, 0x3100, 0x1c11, 0xb284, 0x0024, 0x0030, /* 1290 */ 
   0x0490, 0x3800, 0x8024, 0x0025, 0xffd0, 0x3980, 0x1810, 0x36f4, /* 1298 */ 
   0x7c11, 0x36f0, 0x1802, 0x0030, 0x0717, 0x3602, 0x8024, 0x2100, /* 12a0 */ 
   0x0000, 0x3f05, 0xdbd7, 0x0007, 0x0001, 0x82f7, 0x0006, 0x0012, /* 12a8 */ 
   0x0002, 0x4b57, 0x3613, 0x0024, 0x3e00, 0x3801, 0xf400, 0x55c0, /* 12b0 */ 
   0x0000, 0x0897, 0xf400, 0x57c0, 0x0000, 0x0024, 0x2000, 0x0000, /* 12b8 */ 
   0x36f0, 0x1801, 0x0007, 0x0001, 0x8948, 0x0006, 0x0006, 0x2a08, /* 12c0 */ 
   0x1b8e, 0x2800, 0xbe00, 0x0002, 0x5217, 0x0007, 0x0001, 0x894b, /* 12c8 */ 
   0x0006, 0x0028, 0x3e22, 0xb815, 0x3e05, 0xb814, 0x3615, 0x0024, /* 12d0 */ 
   0x0000, 0x800a, 0x4283, 0xe1e3, 0x003f, 0xfe03, 0xa034, 0x0024, /* 12d8 */ 
   0x0000, 0x3fc3, 0xd214, 0x0024, 0x0000, 0x0201, 0xb234, 0x0024, /* 12e0 */ 
   0x0006, 0xc083, 0x4326, 0x0024, 0xa010, 0x40d5, 0x3500, 0x5803, /* 12e8 */ 
   0xd010, 0x1bc1, 0x3405, 0x9014, 0x36e3, 0x0024, 0x2000, 0x0000, /* 12f0 */ 
   0x36f2, 0x9815, 0x0007, 0x0001, 0x895f, 0x0006, 0x001c, 0x3613, /* 12f8 */ 
   0x0024, 0x3e22, 0xb815, 0x3e05, 0xb814, 0x3615, 0x0024, 0x3e00, /* 1300 */ 
   0x3810, 0x0007, 0x0090, 0x34d3, 0x0024, 0x3430, 0x0024, 0x3800, /* 1308 */ 
   0x1810, 0x36f0, 0x0024, 0x3405, 0x9014, 0x36e3, 0x0024, 0x2000, /* 1310 */ 
   0x0000, 0x36f2, 0x9815, 0x0007, 0x0001, 0x896d, 0x0006, 0x0018, /* 1318 */ 
   0x3613, 0x0024, 0x3e22, 0xb815, 0x3e05, 0xb814, 0x3615, 0x0024, /* 1320 */ 
   0x3e04, 0x0024, 0x0007, 0x0090, 0x3000, 0x0024, 0x36f4, 0x0024, /* 1328 */ 
   0x3405, 0x9014, 0x36e3, 0x0024, 0x2000, 0x0000, 0x36f2, 0x9815, /* 1330 */ 
   0x0007, 0x0001, 0x8979, 0x0006, 0x0074, 0x3613, 0x0024, 0x3e22, /* 1338 */ 
   0xb815, 0x3e05, 0xb814, 0x3615, 0x0024, 0x0000, 0x800a, 0x3e10, /* 1340 */ 
   0x3801, 0x3e10, 0xb803, 0xb884, 0xb805, 0x3e11, 0x0024, 0x3e11, /* 1348 */ 
   0xb80d, 0x2802, 0x6940, 0x3e03, 0xf80e, 0xf400, 0x4083, 0x2402, /* 1350 */ 
   0x66ce, 0xf400, 0x4105, 0x0000, 0x0206, 0xa562, 0x0024, 0x455a, /* 1358 */ 
   0x0024, 0x0020, 0x0006, 0xd312, 0x0024, 0xb16c, 0x0024, 0x0020, /* 1360 */ 
   0x0006, 0x2802, 0x6545, 0xd342, 0x0024, 0x0000, 0x01c6, 0xd342, /* 1368 */ 
   0x0024, 0xd56a, 0x0024, 0x0020, 0x0006, 0x4448, 0x0024, 0xb16c, /* 1370 */ 
   0x0024, 0x0020, 0x0146, 0x2802, 0x66c5, 0x0000, 0x0024, 0xd468, /* 1378 */ 
   0x0024, 0x4336, 0x0024, 0x0000, 0x3fc1, 0x0006, 0x8086, 0x462c, /* 1380 */ 
   0x0024, 0xb512, 0x4195, 0x3d00, 0x4024, 0x0006, 0xc081, 0x4122, /* 1388 */ 
   0x0024, 0x6294, 0x4055, 0x3d01, 0x0024, 0x0000, 0x4000, 0x6200, /* 1390 */ 
   0x0024, 0x0000, 0x03ce, 0x2802, 0x6151, 0xb888, 0x0024, 0x36f3, /* 1398 */ 
   0xd80e, 0x36f1, 0x980d, 0x36f1, 0x1805, 0x36f0, 0x9803, 0x36f0, /* 13a0 */ 
   0x1801, 0x3405, 0x9014, 0x36e3, 0x0024, 0x2000, 0x0000, 0x36f2, /* 13a8 */ 
   0x9815, 0x0007, 0x0001, 0x89b3, 0x0006, 0x004c, 0x3613, 0x0024, /* 13b0 */ 
   0x3e12, 0xb817, 0x3e12, 0x3815, 0x3e05, 0xb814, 0x3615, 0x0024, /* 13b8 */ 
   0x0000, 0x800a, 0x3e10, 0x3801, 0x3e10, 0xb810, 0x0006, 0x8a10, /* 13c0 */ 
   0xb880, 0x3811, 0x0006, 0x9fd1, 0x3009, 0x2000, 0x3009, 0x0400, /* 13c8 */ 
   0x4080, 0x0024, 0x0006, 0x8011, 0x2802, 0x7185, 0xbc82, 0x0024, /* 13d0 */ 
   0x0000, 0x0200, 0xbc82, 0xa000, 0x3910, 0x0024, 0x2902, 0x5e40, /* 13d8 */ 
   0x39f0, 0x4024, 0x0006, 0x9f90, 0x0006, 0x9f51, 0x3009, 0x0000, /* 13e0 */ 
   0x3009, 0x0401, 0x6014, 0x0024, 0x0000, 0x0024, 0x2902, 0x7645, /* 13e8 */ 
   0x0002, 0x7248, 0x36f4, 0x4024, 0x36f0, 0x9810, 0x36f0, 0x1801, /* 13f0 */ 
   0x3405, 0x9014, 0x36f3, 0x0024, 0x36f2, 0x1815, 0x2000, 0x0000, /* 13f8 */ 
   0x36f2, 0x9817, 0x0007, 0x0001, 0x89d9, 0x0006, 0x001a, 0x3613, /* 1400 */ 
   0x0024, 0x3e12, 0xb817, 0x3e12, 0x3815, 0x3e05, 0xb814, 0x290a, /* 1408 */ 
   0xd900, 0x3605, 0x0024, 0x2910, 0x0180, 0x3613, 0x0024, 0x3405, /* 1410 */ 
   0x9014, 0x36f3, 0x0024, 0x36f2, 0x1815, 0x2000, 0x0000, 0x36f2, /* 1418 */ 
   0x9817, 0x0007, 0x0001, 0x89e6, 0x0006, 0x0066, 0x3613, 0x0024, /* 1420 */ 
   0x3e12, 0xb817, 0x3e12, 0x3815, 0x3e05, 0xb814, 0x3625, 0x0024, /* 1428 */ 
   0x0000, 0x800a, 0x3e10, 0xb803, 0x0001, 0x0002, 0x3e11, 0x3805, /* 1430 */ 
   0x3e11, 0xb807, 0x3e14, 0x3811, 0x0006, 0x9f90, 0x3e04, 0xb813, /* 1438 */ 
   0x3413, 0x0000, 0x6090, 0x0024, 0x6026, 0x33c0, 0x0000, 0x0024, /* 1440 */ 
   0x2802, 0x7f15, 0x0000, 0x0024, 0x0000, 0x0011, 0x3413, 0x0024, /* 1448 */ 
   0x3cf4, 0x4024, 0x0006, 0x9f50, 0x3413, 0x0002, 0x34f0, 0x0024, /* 1450 */ 
   0x6026, 0x0024, 0x0000, 0x0024, 0x2902, 0x7645, 0x0002, 0x7f08, /* 1458 */ 
   0x0006, 0x9f92, 0x0006, 0x8a13, 0x3413, 0x0811, 0xb58b, 0x1630, /* 1460 */ 
   0x468e, 0x13d1, 0xb68c, 0x0404, 0xcfee, 0x9813, 0x0000, 0x0804, /* 1468 */ 
   0xaf0e, 0x1bd2, 0x4f86, 0x9811, 0x0000, 0x0021, 0x6418, 0x1bd0, /* 1470 */ 
   0x6848, 0x9807, 0xad46, 0x1bc6, 0x4280, 0x9805, 0x36f1, 0x0024, /* 1478 */ 
   0x36f0, 0x9803, 0x3405, 0x9014, 0x36f3, 0x0024, 0x36f2, 0x1815, /* 1480 */ 
   0x2000, 0x0000, 0x36f2, 0x9817, 0x0007, 0x0001, 0x8a19, 0x0006, /* 1488 */ 
   0x0088, 0x3613, 0x0024, 0x3e12, 0xb817, 0x3e12, 0x3815, 0x3e05, /* 1490 */ 
   0xb814, 0x3615, 0x0024, 0x0000, 0x800a, 0x3e10, 0x3801, 0x3e10, /* 1498 */ 
   0xb803, 0x3e11, 0x3805, 0x3e14, 0x3811, 0x2802, 0x91c0, 0x3e04, /* 14a0 */ 
   0xb813, 0x0006, 0x9f90, 0x0001, 0x0001, 0x3009, 0x0000, 0x6090, /* 14a8 */ 
   0x0024, 0x6014, 0xa000, 0x0001, 0x0001, 0x2802, 0x8c11, 0x0000, /* 14b0 */ 
   0x0024, 0x3009, 0x0000, 0x6012, 0x0024, 0x3009, 0x2001, 0x0006, /* 14b8 */ 
   0x9f90, 0x0006, 0x9f51, 0x3009, 0x0000, 0x3009, 0x0401, 0x6014, /* 14c0 */ 
   0x0024, 0x0000, 0x0024, 0x2902, 0x7645, 0x0002, 0x8c08, 0x0000, /* 14c8 */ 
   0x0080, 0x0030, 0x0013, 0xb882, 0x130c, 0xf400, 0x4512, 0x3210, /* 14d0 */ 
   0x910c, 0x32f0, 0xc024, 0x6dc2, 0x0024, 0x3a10, 0x0024, 0x3af0, /* 14d8 */ 
   0x4024, 0x0000, 0x0201, 0x3300, 0x0024, 0xb010, 0x0024, 0x0000, /* 14e0 */ 
   0x0024, 0x2802, 0x94d5, 0x0000, 0x0024, 0x6894, 0x130c, 0xb886, /* 14e8 */ 
   0x1040, 0x3430, 0x4024, 0x6dca, 0x0024, 0x0000, 0x0024, 0x2802, /* 14f0 */ 
   0x8951, 0xbcd2, 0x0024, 0x0000, 0x0201, 0x2802, 0x94c5, 0x0000, /* 14f8 */ 
   0x0024, 0x2900, 0xb680, 0x3613, 0x0024, 0x36f4, 0x9813, 0x36f4, /* 1500 */ 
   0x1811, 0x36f1, 0x1805, 0x36f0, 0x9803, 0x36f0, 0x1801, 0x3405, /* 1508 */ 
   0x9014, 0x36f3, 0x0024, 0x36f2, 0x1815, 0x2000, 0x0000, 0x36f2, /* 1510 */ 
   0x9817, 0x0007, 0x0001, 0x8a5d, 0x0006, 0x0018, 0x4080, 0x184c, /* 1518 */ 
   0x3e13, 0x780f, 0x2802, 0x9945, 0x4090, 0xb80e, 0x2402, 0x98c0, /* 1520 */ 
   0x3e04, 0x0440, 0x3810, 0x0440, 0x3604, 0x0024, 0x3009, 0x1bce, /* 1528 */ 
   0x3603, 0x5bcf, 0x2000, 0x0000, 0x0000, 0x0024, 0x000a, 0x0001, /* 1530 */ 
   0x0300 
 }; 

#endif

