/* mbed Microcontroller Library
 * Copyright (c) 2015-2016 Nuvoton
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */


 #include "UART9BIT.h"
 #include "serial_api9bit.h"


UART9BIT::UART9BIT(PinName tx, PinName rx, int baud,bool en9bit) :_thunk9_irq(this), Serial( tx,  rx,  baud)
{
 Enable9Bit=en9bit;
 if(en9bit)
 format(9,UART9BIT::None,1);
}

void UART9BIT::format(int bits, Parity parity, int stop_bits) 
{
    lock();
    serial_format_9bit(&_serial, bits, (SerialParity)parity, stop_bits);
    unlock();
}


void UART9BIT::interrupt_handler_asynch(void)
{
  
    int event;    
    int rx_event;

     

    if(Enable9Bit)
    {
    event=serial_irq_handler_asynch9bit(&_serial);
    
    sprintf(uart9debug,"Uart9 serial_irq_handler_asynch9bit event=%04x\r\n",event);  uartdebug=true;

    }
    else
    {
    event=serial_irq_handler_asynch(&_serial);   
    }

    rx_event = event & SERIAL_EVENT_RX_MASK;



    if (_rx_asynch_set && rx_event) {
        

        event_callback_t cb = _rx_callback;
        _rx_asynch_set = false;
        _rx_callback = NULL;

        if (cb) {
             sprintf(uart9debug,"Uart9 run cb \r\n");
         uartdebug=true;            
            cb.call(rx_event);
        }
        sleep_manager_unlock_deep_sleep();
    }

    int tx_event = event & SERIAL_EVENT_TX_MASK;
    if (_tx_asynch_set && tx_event) {
        event_callback_t cb = _tx_callback;
        _tx_asynch_set = false;
        _tx_callback = NULL;
        if (cb) {
            cb.call(tx_event);
        }
        sleep_manager_unlock_deep_sleep();
    }
}

int UART9BIT::write(const uint16_t *buffer, int length, const event_callback_t &callback, int event)
{
    int result = 0;   
    lock();
  
    if (!serial_tx_active(&_serial) && !_tx_asynch_set) {
        start_write((void *)buffer, length, 16, callback, event);
    } else {
        result = -1; // transaction ongoing
    }


    unlock();
     
    return result;
}


void UART9BIT::start_write(const void *buffer, int buffer_size, char buffer_width, const event_callback_t &callback, int event)
{
    _tx_asynch_set = true;
    _tx_callback = callback;
    
    _thunk9_irq.callback(&UART9BIT::interrupt_handler_asynch);

    sleep_manager_lock_deep_sleep();
     
    if(Enable9Bit)
    {
    serial_tx_asynch9bit(&_serial, buffer, buffer_size, buffer_width, _thunk9_irq.entry(), event, _tx_usage);
    }
    else
    {
      serial_tx_asynch(&_serial, buffer, buffer_size, buffer_width, _thunk9_irq.entry(), event, _tx_usage);    
    }
     
}


void UART9BIT::start_read(void *buffer, int buffer_size, char buffer_width, const event_callback_t &callback, int event, unsigned char char_match)
{
    _rx_asynch_set = true;
    _rx_callback = callback;

  
    _thunk9_irq.callback(&UART9BIT::interrupt_handler_asynch);
    sleep_manager_lock_deep_sleep();

    if(Enable9Bit)
    {
    serial_rx_asynch9bit(&_serial, buffer, buffer_size, buffer_width, _thunk9_irq.entry(), event, char_match, _rx_usage);   
    }
    else
    {    
    serial_rx_asynch(&_serial, buffer, buffer_size, buffer_width, _thunk9_irq.entry(), event, char_match, _rx_usage);
    }

}

uint16_t UART9BIT::getu16()
{
  return ::serial_getc_9bit(&_serial);
}

int UART9BIT::serial_readable(void)
{
  return ::serial_readable_9bit(&_serial);
}

void UART9BIT::putu16(uint16_t data)
{
  return ::serial_putc_9bit(&_serial,data);
}

int UART9BIT::read(uint16_t *buffer, int length, const event_callback_t &callback, int event, unsigned char char_match)
{
    int result = 0;
    lock();
    if (!serial_rx_active(&_serial) && !_rx_asynch_set) {        
        start_read((void *)buffer, length, 8, callback, event, char_match);
    } else {
        result = -1; // transaction ongoing
    }
    unlock();
    return result;
}

int UART9BIT::GitRxDataLen(void)
{
  int retcnt=0;
  retcnt=_serial.rx_buff.pos;
  
  return retcnt;
}

void UART9BIT::SetRxDataLen(size_t len)
{
  int retcnt=0;
  _serial.rx_buff.pos=len;
  

}



void UART9BIT::GetUartName(void)
{
    switch(_serial.serial.uart)
    {
        case UART_0:
            printf("Uart 0 init \r\n");
        break;
          case UART_1:
            printf("Uart 1 init \r\n");
        break;
          case UART_2:
            printf("Uart 2 init \r\n");
        break;
          case UART_3:
            printf("Uart 3 init \r\n");
        break;
          case UART_4:
            printf("Uart 4 init \r\n");
        break;
          case UART_5:
            printf("Uart 5 init \r\n");
        break;
    }



}




