#include "TouchPanel.h"

/***************************************************************************//**
 * @brief Creates an instance of the TouchPanel class. Each of the following
 *        pins has to be connected to an AnalogIn pin.
 *
 * @param xPos  X+
 * @param xNeg  X-
 * @param yPos  Y+
 * @param yNeg  Y-
*******************************************************************************/
TouchPanel::TouchPanel(PinName xPos, PinName xNeg, PinName yPos, PinName yNeg)
{
    this->xPos = xPos;
    this->xNeg = xNeg;
    this->yPos = yPos;
    this->yNeg = yNeg;
    this->xy = true;
}

/***************************************************************************//**
 * @brief Calls the PollPanel method in specified intervals.
 *
 * @param touchCallbackFunction     Pointer to the touch callback function.
 *                                  Example prototype:
 * @code void TouchCallback(unsigned short x, unsigned short y); @endcode
 *
 * @return None.
*******************************************************************************/
void TouchPanel::SetTouchCallbackFunction(pTouchCallback_t touchCallbackFunction)
{
    this->touchCallbackFunction = touchCallbackFunction;
    this->PrepareYRead();
    this->pollPanelTicker.attach(this, &TouchPanel::PollPanel, POLLING_INVERVAL);
}

/***************************************************************************//**
 * @brief Prepares the touch panel and the software in order read the
 *        Y coordinate.
 *
 * @return None.
*******************************************************************************/
void TouchPanel::PrepareYRead(void)
{
    delete yPosOut;
    delete yNegOut;
    delete xPosIn;
    delete xNegIn;
    xPosOut = new DigitalOut(xPos);
    xNegOut = new DigitalOut(xNeg);
    xPosOut->write(1);
    xNegOut->write(0);
    yPosIn = new AnalogIn(yPos);
    yNegIn = new AnalogIn(yNeg);
}

/***************************************************************************//**
 * @brief Prepares the touch panel and the software in order read the
 *        X coordinate.
 *
 * @return None.
*******************************************************************************/
void TouchPanel::PrepareXRead(void)
{
    delete xPosOut;
    delete xNegOut;
    delete yPosIn;
    delete yNegIn;
    yPosOut = new DigitalOut(yPos);
    yNegOut = new DigitalOut(yNeg);
    yPosOut->write(1);
    yNegOut->write(0);
    xPosIn = new AnalogIn(xPos);
    xNegIn = new AnalogIn(xNeg);
}

/***************************************************************************//**
 * @brief Alternately reads the touch panel's X or Y coordinate and calls the
 *        touch callback function when a touch was detected.
 *
 * @return None.
*******************************************************************************/
void TouchPanel::PollPanel(void)
{
    static float yp, yn, xp, xn;
    static unsigned short x = NO_TOUCH_VAL;
    static unsigned short y = NO_TOUCH_VAL;
    static bool newTouch = true;

    if(this->xy) {
        yp = yPosIn->read();
        yn = yNegIn->read();
        this->PrepareXRead();
        if(fabs(yp - yn) < TOUCH_THRESHOLD) {
            yp = HEIGHT * (1 - (yp - BOTTOM) / (TOP - BOTTOM));
            if(yp < 0) {
                y = 0;
            } else if(yp > HEIGHT) {
                y = HEIGHT;
            } else {
                y = yp;
            }
        } else {
            y = NO_TOUCH_VAL;
            newTouch = true;
        }
    } else {
        xp = xPosIn->read();
        xn = xNegIn->read();
        this->PrepareYRead();
        if(fabs(xp - xn) < TOUCH_THRESHOLD) {
            xp = WIDTH * (xp - LEFT) / (RIGHT - LEFT);
            if(xp < 0) {
                x = 0;
            } else if(xp > WIDTH) {
                x = WIDTH;
            } else {
                x = xp;
            }
        } else {
            x = NO_TOUCH_VAL;
            newTouch = true;
        }
    }
    this->xy = !this->xy;
    if(x != NO_TOUCH_VAL && y != NO_TOUCH_VAL && newTouch && x > 0) {
        newTouch = false;
        this->touchCallbackFunction(x, y);
    }
}
