#ifndef MBED_FRENCH_TIME_H
#define MBED_FRENCH_TIME_H

#include "mbed.h"

#define UTC_TIME_OFFSET  1
#define DAYS_IN_MARS  31



/**
 * Classe qui permet de calculer l'heure locale en France à partir de l'heure UTC provenant d'un GPS par exemple.
 * Le passage à l'heure d'été est réalisé le dernier dimanche de mars à 00:00
 * Le passage à l'heure d'hiver est réalisé le dernier dimanche d'octobre à 00:00
 */
class FrenchTime
{

public:

    /** Mois de l'année */
    enum month {JANVIER = 1 /**<Le mois de Janvier = 1*/, FEVRIER /**< 2 */, MARS /**< 3 */, AVRIL /**< 4 */, MAI /**< 5 */, JUIN /**< 6 */, JUILLET /**< 7 */, AOUT /**< 8 */, SEPTEMBRE /**< 9 */, OCTOBRE /**< 10 */, NOVEMBRE /**< 11 */, DECEMBRE /**< 12 */ };

    /** Jours de la semaine */
    enum dayOfWeek {DIM = 0 /**<Dimanche = 0 */, LUN /**< 1 */, MAR /**< 2 */, MER /**< 3 */, JEU /**< 4 */, VEN /**< .5 */, SAM /**<Samedi = 6 */ };


    /**
     * Methode qui revoie le décalage horaire de l'heure francaise par rapport à l'heure UTC
     * @param day date du jour
     * @param month mois
     * @param year année
     * @returns
     *  uint8_t 1 (hiver)
     *  uint8_t 2 (été)
     *
     * @code
     *  int DecalageHoraire = FrenchTime::frenchTimeOffset(29,10,2016); //return 1
     *  int DecalageHoraire = FrenchTime::frenchTimeOffset(30,10,2016); //return 2 
     * @endcode
     */
    static uint8_t frenchTimeOffset(uint8_t day, uint8_t month, int year);


    /**
     * Methode qui renvoie la date du dernier dimanche d'un mois
     * @param month mois
     * @param year année
     * @return uin8_t jour du mois de 1 à 31
     *
     * @code
     *  int dateDuDernierDimancheDuMois = FrenchTime::lastSundayOfMonth(5,2016); //return 29
     * @endcode
     */
    static uint8_t lastSundayOfMonth(uint8_t month, int year);


    /**
     * Methode qui renvoie le jour de la semaine correspondant à une date
     * @param day date du jour
     * @param month mois
     * @param year année
     * @return uin8_t jour de la semaine de 0(dimanche) à 6(samedi)
     *
     * @code
     *  int jour = FrenchTime::dayOfWeek(22,6,2016); //return 3
     * @endcode
     */
    static uint8_t dayOfWeek(uint8_t day, uint8_t month, int year);
};

#endif
