
/**
@file main.h
@brief header file containing function prototypes, defines and variables
@brief shows the different applications of the ultrasonic distance sensor combined with a temperature sensor
@brief Last Revision combining all of the previous trials
@author Mahlet Shimellis
@date April-March 2015
*/

#ifndef MAIN_H
#define MAIN_H

/**
@brief Below is the list of temperature sensor register adresses used
@brief the first two are read and write adresses when ADD0 is connected to GROUND
@brief the last two are common temerature registers
*/

#define TMP102_R_ADD 0x91
#define TMP102_W_ADD 0x90
#define TEMP_REG 0x00
#define CONFIG_REG 0x01

#include "mbed.h"

/**
@namespace leds
@brief This defines the mbed leds taht will be useful to detect error messages
*/
BusOut leds(LED4,LED3,LED2,LED1);

/**
@namespace lcd
@brief  LCD dispaly pin connections to the mbed
@brief Format used N5110(PinName pwrPin, PinName scePin, PinName rstPin, PinName dcPin, PinName mosiPin, PinName sclkPin, PinName ledPin)
*/
N5110 lcd(p5,p6,p7,p8,p11,p13,p21);

/**
@namespace buzzer
@brief Buzzer connection to mbed
*/
Beep buzzer(p26);

/**
@namespace button
@brief anolog input to the mbed
*/
InterruptIn button(p16);

/**
@namespace srf
@brief SRF02 ultrasonic distance sensor I2C connection to mbed
@brief Format srf(SDA pin, SCL pin)
*/
SRF02 srf(p28, p27);

/**
@namespace tmp102
@brief TMP102 temperature sensor I2C connection to mbed
@brief format tmp102(SDA pin, SCL pin)
*/
//I2C tmp102(p9,p10);

/**
@namespace red
@brief Red  LED PwmOut connection to mbed
*/
PwmOut red(p22);

/**
@namespace yellow
@brief Yellow LED PwmOut connection to mbed
*/
PwmOut yellow(p23);

/**
@namespace green
@brief Green LED PwmOut connection to mbed
*/
PwmOut green(p24);



Ticker timer; ///global timer
Ticker measure_ticker;
Timer measure;
int measureFlag=0; /*!< measure flag set in ISR for intial temperature and distance measurements  ??? ticker timer */
double distance; /*!< distance value updated from SRF02 measurements */
float temperature; /*!<temperature value updated from TMP102 measurements*/
float soundSpeed; /*!<float for speed of sound */
float soundTimeDelay; /*!<float time delay value for sending and receiving sound signal */
Ticker temperature_ticker;

/**
@brief defines the high flag state for the measurement flag
*/
void measureExpired ();

/**
@bSets the error signaling for the mbed connections mainly I2C when the acknowlegment bit is not received
@param code - integer to classify error type
*/
void error(int code);




/**
Initialises the temperature sensor
*/
void initTMP102();

/**
Performs the temperature measurement
@returns the temperature value with a 0.0625 resolution
*/
float getTemperature();


/**
Calculates the speed of sound from the measured temperature
@param temperature - float measured temperature value
@returns the value of the speed of sound for each specific temeprature value
*/
 float calcSoundSpeed (float temperature);


/**
Calculates the delay between transmission and emission of the sound from the ultrasonic distance sensor
@param distance - integer measured distance value
@param soundSpeed - float calculated speed of sound value
@returns timeDelay - float delay between transmission and reception of the sound after reflecting on the object for which the distance has been measured
*/
float calcSoundTimeDelay (int distance, float soundSpeed);



/**
////////////////////PART 2 - Distance Update and Speed ///////////////////////////////////
@brief in this section
@ brief variables
@brief ticker variable
@brief timer expired function   
@brief distance update and storage function
*/


Ticker update_ticker;
Ticker update_ticker_two;
int updateFlag=0;
int r=0;
float newDistance; /*!< new distance value updated from SRF02 measurements after each timer delay */
float lastDistance;/*!< for eaxh new distance value updated from SRF02 measurements after each timer delay, this is the stored previous distance value */
float updateDelay=0; /*!< time delay between each distance value update */

///Ticker update_ticker;
//Ticker timer;

/**
Defines the high flag state for the distance measurement update flag
*/
void upadteExpired();

/**
Stores the previous distance in the lastDistance variable and updates the newDistance variable
@returns the lastDistance value of the previous measurement
@returns the newDistance value from the new measurement
*/
void update_distance_new();



/**
/////////////////////////////////////////PART 3 - Distance Range Alert///////////////////////////////////////////
@brief in this section
@brief define min, max, maxState
@brief define variables (state variables, buzzerFrequency)
@brief define updateState function protoype
*/

#define MAX 200
#define MIN 20
#define maxState 4

/**
@brief Timing elements for the range differentiation module of the distance sensor
*/
Ticker range_ticker; 
Timer range;
int rangeFlag =0; /*!< range flag set in ISR */

int currentState =0; /*!<integer from 0 to 4 that defines the current state of the FSM*/
int rangeState =0; /*!<integer from 0 to 4 that defines the range to which belongs the measured distance between MIN and MAX*/

float buzzerFrequency =0; /*!<float that defines the frequency of the buzzer depending on the distance range. higher frequencies as we get closer to the object. output of the FSM structure*/

/**
@brief defines the high flag state for the measurement flag
*/
void rangeExpired();

/**
 function that defines the input of the FSM depending on the distance range from MAX to MIN
@param distance - integer measured value by the SRF02 sensor and to be classified as part of a set of five ranges from MAX to MIN
@returns rangeState - integer from 0 to 4. defined depending on the distance range in whcih the measured distance belongs
@returns currentState - integer from 0 to 4. depending on the value of the rangeState, the actual current state and the FSM nextState from the structure definition
*/
void updateState();



/**
@brief PART 4 - Beam Pattern hard
@brief in this section
@brief define variables (angle related variables, counter, distance related variables)
@brief define trigonometric function protoptypes
@brief define drawing function prototypes
 */

Ticker beam_ticker;
Timer beam;
int beamFlag=0;

/**
@brief defines the high flag state for the beam flag
*/
void beamExpired();

int radius =0;

double angleDegrees =0; /*!<initial angles in degrees*/
double angleRadians=0; /*!<angles in radians after conversion*/
double a=0;
int z=0; ///counter
double d=0;
float lateralDistance=0;
float h=0; ///hypothenuse


/**
@brief function that converts angle units 
@param angleDegrees - double
@returns angleRadians
*/
double convertDegrees (double angleDegrees);

/**
 function that draws a semi-circle depending on the distance measured
@param radius - equal to the disance measured
@brief center at (WIDTH/2,1) on the LCD 
*/
void drawSemiCircle();

/**
 function that shifts through angles of the beam pattern
@param l- lateral distance for angle a
@param h - hypothenuse distance for angle a
*/
void drawBeamLines(float lateralDistance, float h);

/**
 function that calculates the lateral distance between the sensor center and object at angle a
@param a - double angle between center if sensor and object detected
@param d - double shortest right angle distance between center of center and object detected. this is the distance measured directly in front of the sensor
@returns - value of the lateral distance between sensor center line and beam at angle a from center
*/
double lat(double a, double d);

/**
@brief function that calculates the hypothenuse distance between the sensor center and object at angle a - this is the longest distance
@param a - double angle between center if sensor and object detected
@param d - double shortest right angle distance between center of center and object detected. this is the distance measured directly in front of the sensor
@returns - value of the hypothenuse between sensor center line and beam at angle a from center
*/
double hyp(double a, double d);



/**
///////////////////////////////// PART 5 - the Wall Follower Simulation///////////////////////////
@brief in this section
@brief buzzer and led tickers
@brief buzzer functions prototype
@brief led functions prototype
@brief counter integer variables
@breif movemetn and drawing functions prototype
*/


int wallFlag=0;
Timer wall_timer;
Ticker wall_ticker;
/**
Defines the high flag state for the beam flag
*/
void wallExpired();

/**
@brief set of buzzer tickers depending on the direction of movement and signal given 
@brief zero move forward
@brief one turn left
@brief two turn left and move forward
@brief three turn right
@brief four turn right and move forward
@brief five stop, dead end
*/
Ticker beep_ticker_zero;
Ticker beep_ticker_one;
Ticker beep_ticker_two;
Ticker beep_ticker_three;
Ticker beep_ticker_four;
Ticker beep_ticker_five;


/**
@brief set of LED tickers
*/
Ticker toggle_green_ticker;
Ticker toggle_yellow_ticker;
Ticker toggle_red_ticker;


/**
@brief Set of integer pixel counters to define and kepp track of the movement along the maze
*/
int k=0; /*!<integer for stage 0 move forward k from 0 to 22 pixel points*/
int l=0;/*!<integer for stage 1 turn left and move forward one l from 0 to 64 pixel points*/
int m=0;/*!<integer for stage 2 turn left and move forward two m from 0 to 16 pixel points*/
int n=0;/*!<integer for stage 3 turn right and move forward n from 0 to 78 pixel points*/
int pix=0;/*!<integer that keeps track of the sum of all the counter integers as the pixel moves along the maze. helps keep track of the location of the pixel */


int buttonCounter; /*!<integer 0 to 4 counts the number of times the button has been pressed*/
int buttonFlag = 0;/*!<button flag set in ISR*/

/**
Manages the order of displays on the screen and synchronises it to the number of button presses
*/
void manageScreen();

/**
Counts the pixel step in the forward movement 
@param k - integer counter from 0 to 22
@returns pix - integer counter incremented stores values from 0 to k
*/
void moveForward();

/**
Counts the pixel step in the turnLeftForward_one movement 
@param l - integer counter from 0 to 64
@returns pix - integer counter incremented stores values from k to k+l
*/
void turnLeftForward_one();

/**
Counts the pixel step in the turnLeftForward_two movement 
@param m - integer counter from 0 to 16
@returns pix - integer counter incremented stores values from k+l to k+l+m
*/
void turnLeftForward_two();

/**
Counts the pixel step in the turnLeftForward_two movement 
@param n - integer counter from 0 to 78
@returns pix - integer counter incremented stores values from k+l+m to k+l+m+n
*/
void turnRightForward();

/**
Controls intial beep zero move forward
@param sound_zero - integer set frequency value, introduced as local variable for function
@param time_zero - float set time value for each buzz, introduced as local variable for function
*/
void beep_zero();

/**
Turns on beep one Turn Left
@param sound_one - integer set frequency value , local variable for function
@param time_one - float set time value for each buzz , local variable for function
*/
void beep_one();

/**
Turns on beep two Trun Left and move Forward 
@param sound_two - integer set frequency value, local variable for function
@param time_two - float set time value for each buzz, local variable for function
*/
void beep_two();

/**
Turns on beep three Turn Right
@param sound_three - integer set frequency value, local variable for function
@param time_three - float set time value for each buzz, local variable for function
*/
void beep_three();

/**
Turns on beep four Trun Right and move Forward 
@param sound_four - integer set frequency value, local variable for function
@param time_four - float set time value for each buzz, local variable for function
*/
void beep_four();

/**
Turns on beep five Stop Dead End
@param sound_five - integer set frequency value, local variable for function
@param time_five - float set time value for each buzz, local variable for function
*/
void beep_five();


/**
@brief funciton that toggles Red LED
@brief controls PwmOut pin
*/
void toggle_red();

/**
@brief funciton that toggles Yellow LED
@brief controls PwmOut pin
*/
void toggle_yellow();

/**
@brief funciton that toggles Green LED
@brief controls PwmOut pin
*/
void toggle_green();

/**
Controls the pixel movement along the maze
Combines all of the different stages/sub-movements
Combines the buzzer and led tickers for appropriate and timely signaling
@param pix - integer that sums the movement cnouters and gives an idea of the location of the pixel
@returns buzzer sound - output sound signal with frequency depending on the location
@returns LED light - output light signal with color dependent on message/location (go forward, stop etc.)
*/
void move();

/**
Draws the sample structure of the maze to reflect the wall follower movement technique
Combines drawRect functions with pre-set dimensions
@param x,y, width, height
*/
void drawMaze();



#endif
