#include "mbed.h"
#include "TextLCD.h"

#define ENABLED                     1
#define DISABLED                    0

#define ENABLE_SERIAL_OUTPUT        ENABLED
#define ENABLE_DISPLAY              ENABLED

#define DEBOUNCE_TIME_MS            50
#define POTMETER_SAMPLES            10
#define MIN_WELDING_TIME_MS         40
#define WELDING_TIME_MS             3000
#define POTMETER_UPDATE_TIME_MS     250

#if ENABLE_SERIAL_OUTPUT
Serial pc(USBTX, USBRX);
#endif

DigitalOut myled(LED1);
DigitalIn pedal(PB_6);
AnalogIn pot(PA_0);
DigitalOut relay(PB_0);

#if ENABLE_DISPLAY
TextLCD lcd(PB_1, PF_1, PA_8, PA_11, PB_5, PB_4); // rs, e, d4-d7
DigitalOut rw(PF_0);
#endif

bool bActive = false;
unsigned int runTime = 0;
Ticker potUpdateTicker;

void update_time(void);
unsigned int get_time(void);

int main()
{    
    // Initialize pedal
    pedal.mode(PullUp);
    
#if ENABLE_SERIAL_OUTPUT
    // Initialize Serial
    pc.baud(115200);
    pc.printf("Spot Welder 3000\r\n");
#endif
    
    // Initialize time
    get_time();
    
    // Initialize relay output
    relay = 1;
    
#if ENABLE_DISPLAY
    // Initialize LCD
    rw = 0;
    lcd.cls();
    lcd.locate(0,0);
    lcd.printf("Spot Welder 3000");    
    lcd.locate(0,1);
    lcd.printf("Time: %4d ms", runTime);
#endif

    // Start update time
    potUpdateTicker.attach_us(&update_time, POTMETER_UPDATE_TIME_MS * 1000);
    
    while (1)
    {
        // Check for pedal press
        if (pedal == 0)
        {
            bActive = true;
            
            // Debounce
            wait_ms(DEBOUNCE_TIME_MS);
            if (pedal == 0)
            {
#if ENABLE_SERIAL_OUTPUT
                pc.printf("Welding for %d ms\r\n", runTime);
#endif
                relay = 1;
                myled = 1; // LED is ON   
                wait_ms(runTime);
            }
        }
        
        myled = 0; // LED is OFF
        relay = 0;
        bActive = false;
        
        // One-shot
        while (pedal == 0)
        {
            wait_us(1);
        }
    }
}

void update_time(void)
{
    if (not bActive)
    {
        get_time();
#if ENABLE_SERIAL_OUTPUT
        lcd.locate(6,1);
        lcd.printf("%4d", runTime);
#endif
    }
}

unsigned int get_time(void)
{
    float potSum = 0;
    
    for (uint8_t i = 0; i < POTMETER_SAMPLES; i++)
    {
        potSum += pot.read();
    }
    
    runTime = MIN_WELDING_TIME_MS + (unsigned int)((potSum / POTMETER_SAMPLES) * WELDING_TIME_MS);
    return runTime;
}