/* mbed Microcontroller Library
 * Copyright (c) 2006-2013 ARM Limited
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
#ifndef RPCVARIABLE_H_
#define RPCVARIABLE_H_

#include "rpc.h"

namespace mbed {

/**
 *Class to read and set an attached variable using the RPC
 *
 */
template<class T>
class RPCVariable: public RPC {
public:
    /**
     * Constructor
     *
     *@param ptr Pointer to the variable to make accessible over RPC. Any type of
     *variable can be connected
     *@param name The name of that this object will be over RPC
     */
    template<class A>
    RPCVariable(A * ptr, const char * name) : RPC(name) {
        _ptr = ptr;
    }
    /**
     *Read the variable over RPC.
     *
     *@return The value of the variable
     */
    T read() {
        return (*_ptr);
    }
    /**
     *Write a value to the variable over RPC
     *
     *@param The value to be written to the attached variable.
     */
    void write(T value) {
        *_ptr = value;
    }

    virtual const struct rpc_method *get_rpc_methods();
    static struct rpc_class *get_rpc_class();

private:
    T * _ptr;
};

template<class T>
const rpc_method *RPCVariable<T>::get_rpc_methods() {
    static const rpc_method rpc_methods[] = {
        {"read" , rpc_method_caller<T, RPCVariable, &RPCVariable::read> },
        {"write", rpc_method_caller<RPCVariable, T, &RPCVariable::write> },
        RPC_METHOD_SUPER(RPC)
    };
    return rpc_methods;
}

template<class T>
rpc_class *RPCVariable<T>::get_rpc_class() {
    static const rpc_function funcs[] = {
            "new", rpc_function_caller<const char*, T, const char*, &RPC::construct<RPCVariable, T, const char*> > ,
            RPC_METHOD_END
    };
    static rpc_class c = {"RPCVariable", funcs, NULL};
    return &c;
}


/// Specialisation of the RPCVariable class to handle strings.
/// The class is passed a pointer to the buffer and its size
/// to avoid buffer overruns.
template<>
class RPCVariable<char *>: public RPC {
public:
    /**
     * Constructor
     *
     *@param var Pointer to the char buffer used to hold the string being exposed by RPC.
     *@param bufSize Size of the buffer in chars.
     *@param name The name of that this object will be over RPC
     */
    RPCVariable(char *var, int bufSize, const char * name = NULL) : RPC(name), _var(var), _size(bufSize - 1) {
    }
    
    /**
     *Read the variable over RPC.
     *
     *@return The value of the buffer
     */
    char * read() {
        return (_var);
    }
    /**
     *Write a value to the variable over RPC
     *
     *@param The value to be written to the buffer.
     */
    void write(char *value) {
        strncpy(_var, value, _size);
        _var[_size] = 0;
    }


    virtual const rpc_method *get_rpc_methods() {
        static const rpc_method rpc_methods[] = {
            {"read" , rpc_method_caller<char *, RPCVariable, &RPCVariable::read> },
            {"write", rpc_method_caller<RPCVariable, char *, &RPCVariable::write> },
            RPC_METHOD_END
        };
        return rpc_methods;
    }

private:
    char *_var;
    
    // Holds the number of chars excluding the terminating null.
    int _size;
};

}

#endif  //RPCVARIABLE_H_
