//****************************************************************************
// Zak Gamache and Marcus Quettan
//
//Drum Machine
//      - Write tablature based on input from user playing on a drum pad
//      - Playback sound relating to user input
//****************************************************************************

#include "mbed.h"
#include "MSCFileSystem.h"

#define FSNAME "msc"
MSCFileSystem msc(FSNAME);

// Unused AnalogIn inputs set as DigitalIn to reduce A/D noise
DigitalOut noise1(p16);
DigitalOut noise2(p17);
DigitalOut noise3(p18);
DigitalOut noise4(p19);
DigitalOut noise5(p20);

DigitalOut led1(LED1);      // Hit strength level 1
DigitalOut led2(LED2);      // Hit strength level 2
DigitalOut led3(LED3);      // Hit strength level 3
DigitalOut led4(LED4);      // Hit strength level 4

DigitalIn write(p8);        // USB Write Control: Push - Stop Write
DigitalIn sound(p9);        // Switch to enable playback of sound
PwmOut driver(p26);         // PWM to control sound volume
AnalogIn piezo(p15);        // Input from vibration sensor
Serial pc(USBTX, USBRX);    // mbed Serial over USB
Ticker PrintMe;             // Used to print tablature at a regular interval

int flag;                   // Hit tracker: 1 - Hit, 0 - No hit
int count = 0;              // Separates time frames
float voltage;              // Voltage from piezoelectric vibration sensor
int writeNow = 0;           // Write every so often


// File I/O for tablature
FILE *tabFile = fopen( "/" FSNAME "/tablature.txt", "w");

//****************************************************************************
// Keep (musical) time and tell program to write
//****************************************************************************
void printNote()
{
    // Tell the program to write
    writeNow = 1;
}

//****************************************************************************
// Main Program
//      - Continuously read vibration sensor and scale to a hit strength
//      - Call ticker actions at a regular interval
//****************************************************************************
int main()
{
    // Flash lights and print error if file not opened
    if ( tabFile == NULL ) {
        error("Could not open file for write\n");
    }

    // Setup pullup resistor for sound switch
    sound.mode(PullUp);
    write.mode(PullUp);
    wait(0.01);

    // Print to the tablature every half second (slow quarter note)
    PrintMe.attach(&printNote, 0.50);

    // Run forever
    while(1) {
        // On switch toggle
        if (!write) {
            fclose(tabFile);        // Close file
            tabFile = NULL;         // Set file pointer to null
            pc.printf("Exit");      // Alert user to exit
            break;
        }
        voltage = piezo;

        // 1 - Softest Hit
        if ((voltage >= 0.495 && voltage < 0.515)) {
            if(flag < 2) {
                flag = 1;
            }
            led1 = 1;
        }
        // 2 - Soft Hit
        else if ((voltage >= 0.515 && voltage < 0.530)) {
            if (flag < 3) {
                flag = 2;
            }
            led2 = 1;
        }
        // 3 - Hard Hit
        else if ((voltage >= 0.530 && voltage < 0.550)) {
            if(flag < 4) {
                flag = 3;
            }
            led3 = 1;
        }
        // 4 - Hardest Hit
        else if ((voltage >= 0.550)) {
            flag = 4;
            led4 = 1;
        }
        // 0 - No Hit
        else {
            led1 = 0;
            led2 = 0;
            led3 = 0;
            led4 = 0;
        }

        if (writeNow) {
            if (tabFile) {
                if (count == 0) {
                    pc.printf("\n\n"
                              "1 e & a 2 e & a 3 e & a 4 e & a || 1 e & a 2 e & a 3 e & a 4 e & a"
                              "\n");
                    fprintf(tabFile, "\n\n"
                            "1 e & a 2 e & a 3 e & a 4 e & a || 1 e & a 2 e & a 3 e & a 4 e & a"
                            "\n", flag);
                }

                // If a hit has occurred
                if (flag > 0) {
                    // Print the strength value of the hit to the tablature
                    if (count == 16) {
                        pc.printf("|| %i ", flag);
                        fprintf(tabFile, "|| %i ", flag);
                    } else {
                        pc.printf("%i ", flag);
                        fprintf(tabFile, "%i ", flag);
                    }

                    // Changes sound output pitch based on strenght of hit
                    if (flag == 1) {
                        driver.period(0.005);
                    } else if (flag == 2) {
                        driver.period(0.0075);
                    } else if (flag == 3) {
                        driver.period(0.01);
                    } else {
                        driver.period(0.015);
                    }

                    // If sound is enabled, set the volume
                    if (sound) {
                        driver = 0.005;
                    } else {
                        driver = 0.0;
                    }

                    // Reset hit tracker and time keeper
                    flag = 0;
                    count += 1;
                    // If no hit has occurred
                } else {
                    // Print an empty note to tablature
                    if (count == 16) {
                        pc.printf("|| - ", flag);
                        fprintf(tabFile, "|| - ", flag);
                    } else {
                        pc.printf("- ", flag);
                        fprintf(tabFile, "- ", flag);
                    }

                    // Make sure the volume is 0 and increment the time keeper
                    driver = 0.0;
                    count += 1;
                }

                // Separates tablature time frames
                if (count == 32) {
                    count = 0;
                }
            } else {
                pc.printf("File Not Open\n");
            }

            // Don't write until next tick
            writeNow = 0;
        }
    }
}