#include "mbed.h"
#include "bc_ant.h"

/** USBTX - Tranmit on USB  USBRX - receive on USB  **/
Serial pc(USBTX, USBRX);

/**  LPC1768 LEDs **/
BusOut myLeds(LED1, LED2, LED3, LED4);

/** for BC-ANT-SERIAL (LPC1768) **/
bc_ant antPort(p13, p14, p15, 9600);

/** DIP-SW (for NODE SHARED ADDRESS)  **/
DigitalIn dip1(p20);   // DIP-SW1
DigitalIn dip2(p19);   // DIP-SW2

/**  OUTPUT DEVICES  **/
DigitalOut led0(p21);
DigitalOut led1(p22);

/**   ANT DEFINES   **/
#define MAX_BUFSIZE             64
#define ANT_CH                   0
#define DEVICE_NUMBER           65
#define ANT_NETWORK_NUMBER       0  // Network Number
#define TRANS_TYPE_SHARED_2BYTES 3  // Shared Channel using 2byte address
#define DEVICE_TYPE_ID           3  // Device Type ID

/** DIP-SWITCH ASSIGN **/
#define ID_1   0x00001   // 1:off, 2:off
#define ID_2   0x00002   // 1:on,  2:off
#define ID_3   0x00004   // 1:off, 2:on
#define ID_4   0x00008   // 1:on,  2:on


/**
 *   CHECK DIP-SW STATUS
 */
int check_DipSw()
{
    if (dip1 == 0)
    {
        return ((dip2 == 0) ? 4 : 2);
    }
    return ((dip2 == 0) ? 3 : 1);
}

/**
 *   GET LED LAMP ID
 */
int getLampId(int id)
{
    int lampId = ID_1;
    switch (id)
    {
      case 4:
        lampId = ID_4;
        break;
      case 3:
        lampId = ID_3;
        break;
      case 2:
        lampId = ID_2;
        break;
      case 1:
      default:
        lampId = ID_1;
        break;
    }
    return (lampId);
}

/**
 *  
 */
void executeAction(uint8_t commandType, uint8_t channel, uint8_t command)
{
    if (commandType != 0x4e)
    {
        // do nothing!
        return;
    }
    // execute!
    if ((command & 0x40) != 0)
    {
        led0 = 1;    
    }
    else
    {
        led0 = 0;
    }
    if ((command & 0x04) != 0)
    {
        led1 = 1;    
    }
    else
    {
        led1 = 0;
    }

}

/**
 *  Recieved a message from ANT port.
 */
void receivedFromAnt()
{
   uint8_t receivedLength = 0;
   uint8_t receivedBuffer[MAX_BUFSIZE];
   receivedLength = antPort.BC_ANT_RecvPacket(receivedBuffer, MAX_BUFSIZE);

   executeAction(receivedBuffer[2], receivedBuffer[4], receivedBuffer[6]);

   // Dump received message to USB
   printf("RX(%3d):", receivedLength);
   for (int index = 0; index < receivedLength; index++)
   {
       printf("%02x ", receivedBuffer[index]);
   }
   printf("\r\n");
}

/**
 *   initializes ANT port
 */
int initialize_ANTport()
{
    antPort.ANT_ResetSystem();
    antPort.ANT_AssignChannel(ANT_CH, ANT_Shared_Bidirectional_Slave, ANT_NETWORK_NUMBER);
    antPort.ANT_SetChannelId(ANT_CH,  DEVICE_NUMBER, DEVICE_TYPE_ID, TRANS_TYPE_SHARED_2BYTES);
    antPort.ANT_SetChannelPeriod_Hz(ANT_CH, 16);   // 16Hz : 32768/16 = 2048
    antPort.ANT_SetChannelRFFreq(ANT_CH, 4);       // 2404 MHz
    antPort.ANT_OpenChannel(ANT_CH);               // 
    antPort.attach(&receivedFromAnt);              // callback function

    // set device's shared address
    uint8_t buffer[MAX_BUFSIZE] =
    {
        0x00, 0x00, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff
    };
    buffer[0] = check_DipSw();
    antPort.ANT_SendBroadcastData(ANT_CH, buffer);
    return (0);
}

/**
 *   Main Routine
 */
int main()
{
    pc.baud(9600);  // set serial speed between PC and mbed.
    printf("--- INITIALIZE (ID:%d)  ----\r\n", check_DipSw());
    int statusLamp = getLampId(check_DipSw());
    initialize_ANTport();      // initialize BC-ANT-SERIAL
    printf("--- READY ----\r\n");

    while(1)
    {
        myLeds = statusLamp;
        wait(0.5);
        myLeds = 0;
        wait(0.5);
    }
}
