/**
 * @section LICENSE
 * Released under the MIT License: http://mbed.org/license/mit
 * Copyright (C) 2012 tosihisa
 *
 * @section DESCRIPTION
 * INA226 - Bi-Directional CURRENT/POWER MONITOR with I2C
 * http://strawberry-linux.com/catalog/items?code=12031
 *
 */
#include "mbed.h"
#include "INA226.hpp"

INA226::INA226(I2C &i2c_,int addr_,int freq_) : i2c(i2c_),i2c_addr(addr_),freq(freq_)
{
    i2c.frequency(freq);
}

int INA226::isExist(void)
{
    char p_addr = 0;   //Select Configuration Register.
    i2c.frequency(freq);
    if(i2c.write(i2c_addr | 0,&p_addr,sizeof(p_addr)) == 0){
        return 1;
    }
    return 0;
}

int INA226::rawWrite(char pointer_addr,unsigned short val_)
{
    char val[3];
    val[0] = pointer_addr;
    val[1] = static_cast<char>((val_ >> 8) & 0x00ff);
    val[2] = static_cast<char>(val_ & 0x00ff);
    i2c.frequency(freq);
    if(i2c.write(i2c_addr | 0,val,sizeof(val)) == 0){
        return 0;
    }
    return 1;
}

int INA226::rawRead(char pointer_addr,unsigned short *val_)
{
    char p_addr = pointer_addr;
    char val[2];
    i2c.frequency(freq);
    if(i2c.write(i2c_addr | 0,&p_addr,sizeof(p_addr)) == 0){
        if(i2c.read(i2c_addr | 0x01,val,sizeof(val)) == 0){
            *val_ = static_cast<unsigned short>(val[0]);
            *val_ = (*val_ << 8) | static_cast<unsigned short>(val[1]);
            return 0;
        }
    }
    return 1;
}

int INA226::getVoltage(double *V_)
{
    unsigned short val;
    if(rawRead(0x02,&val) == 0){
        *V_ = static_cast<double>(val) * 1.25;
        return 0;
    }
    return 1;
}

int INA226::getCurrent(double *I_)
{
    unsigned short val;
    if(rawRead(0x04,&val) == 0){    //INA226は0x04 INA260は0x01
        char *s_p = reinterpret_cast<char *>(&val);
        short d_s;
        char *d_p = reinterpret_cast<char *>(&d_s);
        *(d_p + 0) = *(s_p + 0);
        *(d_p + 1) = *(s_p + 1);
        *I_ = static_cast<double>(d_s) /* 1.25*/;  //INA260では1.25をかける(電流caliblationしないから)
        return 0;
    }
    return 1;
}

int INA226::setCurrentCalibration(unsigned short val)
{
    return rawWrite(0x05,val);  //datasheet(p.21)より0x05はCalibrationResisterへの書き込みを指定している
}
int INA226::setConfigResister(unsigned short config)
{
    return rawWrite(0x00,config);   //0x00でINA226のConfigResisterに指定している(datasheet p.21)
                                    //config(INA226.hで設定)は変換時間と平均回数などの指定をしているデータ
                                    //どちらも大きいほどノイズの影響を受けづらいがデータ取得時間が長くなる。実装システムに合わせて調整。
}
