/*熱電対でのモータ巻き線温度測定、LM61での室温測定、USBメモリへの保存、を行うプログラム*/
/*teratermのシリアルポート設定の通信速度は、115200にしておく(しないと文字化け)*/
/*USBメモリへは、数字が1つ書かれたfileNumberというファイルと、motorTempTest〇(〇にfileNumberの数値が入る)というファイルが生成される。fileNumberは電源ONOFFなどによって値が増えていく*/

#include "mbed.h"
#include "max6675.h"                                //熱電対制御用の設定ファイル
#include "MSCFileSystem.h"                          //USBメモリ制御用の設定ファイル
#include <string>
#include "stdio.h"

Serial pc(USBTX,USBRX);                             //PCとのシリアル通信の設定

FILE *fp;                                           //USBmemoryへ保存するファイルの設定
string fileName = "/usb/motorTempTest";             //USBmemoryに保存するファイルの名前を決定する。この文字列の後に、ファイル切り替え用の番号(fileNumber)や".csv"がつく。(下記プログラムで)
char fileNumber;                                    //ファイル名(番号)切り替え用の変数
MSCFileSystem msc("usb");                           //Mount flash drive under the name "msc" //USBメモリの設定
void USBmemory_init(void);                          //USBメモリの初期設定(初期動作)用関数の宣言
Ticker timer_1s;                                    //1秒ごとに割り込みを発生させるためのタイマ

SPI spi(p11,p12,p13);                               //SPIの通信ポートの設定
max6675 max1(spi,p14);                              //熱電対温度計のSPI通信設定
AnalogIn LM61(p20);                                 //室温測定(LM61)用のA/Dコンバータ(ADC)ポートの設定
DigitalOut P16(p15);                                //使用しないAnalogOutポートをDigitalOutに設定することで、ADCの精度を向上させられる
DigitalOut P17(p16);                                //同上
DigitalOut P18(p17);                                //同上
DigitalOut P19(p18);                                //同上(参考 http://mbed.org/users/chris/notebook/Getting-best-ADC-performance/)
DigitalOut P20(p19);

DigitalIn fileSW(p22);                              //スイッチを押すと、ファイルを新作成するようにする用
DigitalOut measureLED(LED1);                        //測定中にMBEDボードのLED1を光らせる用
DigitalOut saveLED(LED2);                           //保存時ににMBEDボードのLED2を光らせる用

float tempMotor;                                    //モータ温度(熱電対温度)を代入する変数
float tempRoom;                                     //室温(LM61}を代入する変数
bool USBmemory_save = 0;

void handler_timer_1s(){
    saveLED = !saveLED;                                             //LEDのONOFFの状態を反転させる
    USBmemory_save = 1;                                             //USBメモリへの保存フラグを立てる
}

int main() {
    pc.baud(115200);                                                //9600bpsで150ms 115200bpsで13msくらい処理速度が違う(もちろん処理時間は送るデータ量にもよる)
    spi.frequency(4000000);                                         //SPI通信の周波数の設定
    fileSW.mode(PullUp);                                            //fileSW(ファイル切り替えスイッチ)の入力ポートをプルアップに設定
    USBmemory_init();                                               //USBメモリの初期動作確認を行う関数
    wait(1);                                                        //動作開始前に1秒待つ
    timer_1s.attach(&handler_timer_1s, 1.0);
    
    while (1) {
        if( !USBmemory_save ){                                              //USBmemory_saveが0のとき
            measureLED = 1;
            tempMotor = max1.read_temp() - 6;                               //熱電対の温度(モータ巻き線温度)をtempMotorに代入。yokogawaの熱電対計との差(-6℃)を補正している。  
            tempRoom = ((LM61*3.3)-0.600)*100.0;                            //室温(LM61)をtempRoomに代入。計算式はdatasheetを参照
        }
        else{                                                               //USBmemory_saveが1のとき
            fp = fopen( fileName.c_str(), "a");                             //fileName(ファイル名が格納されている文字列)は、USBmemory_init関数で決めている。"a"指定したファイル(fileName)への上書き
                                                                            //fileName.c_str()は、fileNameをstring型からchar型に変換する処理。
            if ( fp == NULL ){                                              //ファイルが開けなかったとき、
                pc.printf("USB ERROR!\r\n");                                //PCにエラー表示する
            }
            fprintf(fp,"%.2f,%.2f\n", tempMotor, tempRoom);                 //モータ温度、室温の順番に書き込む
            fclose(fp);
            pc.printf("Motor:%3.2f Room: %5.2f fileNumber: %d\n\r", tempMotor, tempRoom, fileNumber); //保存したデータをPCに表示する
            USBmemory_save = 0;                                             //USBメモリへの保存フラグをリセットする
        }
        wait(.25);
    }
}



void USBmemory_init(){                                      //USBmemoryに保存するファイルを作る処理        
    fp = fopen( "/usb/fileNumber.txt", "r" );               //ファイル名(番号)を決めるために"fileNumber.txt"内の番号を読み取る。
    if( fp == NULL ){                                       //fileNumber.txtが読み取れなかった場合(USBメモリ内に存在しない場合)、
        FILE *fp1 = fopen( "/usb/fileNumber.txt", "w" );    //fileNumber.txtを新規作成する。
        if( fp1 == NULL ){                                  //新規作成もできなかったとき、
            pc.printf("USB init error\n\r" );               //teratermに設定エラーを表示する
            exit(1);                                        //MBEDボード上のLED1~4を点滅させ、プログラムを終了させる
        }
        fprintf(fp1,"0\n");                                 //新規作成したfileNumber.txtに"0"を書き込む
        pc.printf("create fileNumber.txt!\r\n");            //PCにファイル作成したことを表示する。//timer.start(); //書き込み時間測定開始 //timer.stop(); //書き込み時間測定終了    
        fclose(fp1);                                        //fileNumber.txtを閉じる
        fp = fopen( "/usb/fileNumber.txt", "r" );           //ファイル名(番号)を決めるために"fileNumber.txt"内の番号を読み取る。
    }
    fscanf( fp, "%d",&fileNumber );                         //fileNumber.txt内に書いてある番号を読み取り、fileNumberという変数に代入する
    fclose(fp);                                             //fileNumber.txtを閉じる
    
    fp = fopen( "/usb/fileNumber.txt", "w" );               //usb/fileNumber.txtを開く
    ++fileNumber;                                           //flieNumber.txt内の番号を+1する。(次に生成されるファイルのファイル名(番号)が被らないようにするために)
    fprintf( fp,"%d\n", fileNumber );                       //変えた番号をfileNumber.txtに書き込み
    fclose(fp);                                             //usb/fileNumber.txtを閉じる
    
    char nunu[10] ="0";                                     //fileNumberを数値から文字列に変更するためのchar配列の定義
    sprintf(nunu, "%d", fileNumber);                        //fileNumber(数値)をnunu(文字列)に変換("nunu"は適当)
    fileName += nunu;                                       //fileNameに入ってる文字列(例:"/usb/motorTempTest")に、nunuの文字列(例:1)を連結して代入
    fileName += ".csv";                                     //fileNameに入ってる文字列(例:"/usb/motorTempTest1")に".csv"を連結して代入
    
    fp = fopen( fileName.c_str(), "w");                     //"W"はファイルを新規作成して書き込み,"a"は上書き(Fileなければ新規作成書き込み)
                                                            //fileName.c_str()は、fileNameをstring型からchar型に変換する処理。
    if ( fp == NULL ){                                      //ファイルが開けなかったとき、
        pc.printf("USB ERROR!\r\n");                        //PCにエラー表示する
        exit(1);                                            //MBEDボード上のLED1~4を点滅させ、プログラムを終了させる
    }
    fprintf(fp,"MotorTemp(C),RoomTemp(C)\n");               //ファイル(例:"/usb/motorTempTest1.csv")の1行目に、測定するデータ名を書き込む
    fclose(fp);                                             //ファイル(例:"/usb/motorTempTest1.csv")を閉じる  
    pc.printf("%s\n\r", fileName.c_str());                  //ファイル(例:"/usb/motorTempTest1.csv")名をPCに表示   
}