#include "mbed.h"
#include "security.h"
#include "easy-connect.h"
#include "simple-mbed-client.h"
#include "kx123.h"
#include "BM1383AGLV.h"
 
Serial pc(USBTX, USBRX);
I2C i2c(I2C_SDA, I2C_SCL);
RegisterWriter i2c_rw(i2c);
KX123 acc(i2c_rw, KX123_SECONDARY_SLAVE_ADDRESS);
BM1383AGLV sensor(I2C_SDA, I2C_SCL);

SimpleMbedClient client;

// Declare peripherals
DigitalOut connectivityLed(LED2);                   // Blinks while connecting, turns solid when connected
DigitalOut augmentedLed(LED1, BUILTIN_LED_OFF);     // LED that can be controlled through Connector
InterruptIn btn(MBED_CONF_APP_BUTTON);              // Button that sends it's count to Connector

Ticker connectivityTicker;
Ticker readSensorTicker;
bool connected = false;

// Callback function when the pattern resource is updated
void patternUpdated(string v)
{
    pc.printf("New pattern: %s\n", v.c_str());
}

// Define resources here. They act as normal variables, but are exposed to the internet...
SimpleResourceInt btn_count = client.define_resource("button/0/clicks", 0, M2MBase::GET_ALLOWED);
SimpleResourceString pattern = client.define_resource("led/0/pattern", "500:500:500:500:500:500:500", &patternUpdated);

// Define Rohm sensor module resources
//SimpleResourceInt aX = client.define_resource("accel/0/x", 0, M2MBase::GET_ALLOWED);
//SimpleResourceInt aY = client.define_resource("accel/0/y", 0, M2MBase::GET_ALLOWED);
//SimpleResourceInt aZ = client.define_resource("accel/0/z", 0, M2MBase::GET_ALLOWED);
//SimpleResourceInt press = client.define_resource("press/0/p", 0, M2MBase::GET_ALLOWED);
//SimpleResourceInt temp  = client.define_resource("press/0/t", 0, M2MBase::GET_ALLOWED);
SimpleResourceString accel = client.define_resource("accel/0/xyz", "0,0,0");
SimpleResourceString pressure = client.define_resource("press/0/pt", "0,0");

void readSensor()
{
    char buf[40];
    float res[3];
    
    if (connected == false)
        return;

    acc.getresults_g(res);

//    aX = (int)(res[0] * 10000);
//    aY = (int)(res[1] * 10000);
//    aZ = (int)(res[2] * 10000);
//    press = (int)(sensor.getPressure() * 100);
//    temp = (int)(sensor.getTemperature() * 100);
    
    sprintf(buf, "%f,%f,%f", res[0], res[1], res[2]);
    accel = buf;
    pc.printf("%s\n", buf);
    
    sprintf(buf, "%7.2f,%5.2f", sensor.getPressure(), sensor.getTemperature());
    pressure = buf;
    pc.printf("%s\n", buf);    
}

void fall()
{
    btn_count = btn_count + 1;
    pc.printf("Button count is now %d\r\n", static_cast<int>(btn_count));
}

void toggleConnectivityLed()
{
    connectivityLed = !connectivityLed;
}

void registered()
{
    pc.printf("Registered\r\n");

    connectivityTicker.detach();
    connectivityLed = BUILTIN_LED_ON;
    connected = true;
}

void play(void* args)
{
    connectivityLed = BUILTIN_LED_OFF;

    // Parse the pattern string, and toggle the LED in that pattern
    string s = static_cast<string>(pattern);
    size_t i = 0;
    size_t pos = s.find(':');
    while (pos != string::npos) {
        wait_ms(atoi(s.substr(i, pos - i).c_str()));
        augmentedLed = !augmentedLed;

        i = ++pos;
        pos = s.find(':', pos);

        if (pos == string::npos) {
            wait_ms(atoi(s.substr(i, s.length()).c_str()));
            augmentedLed = !augmentedLed;
        }
    }

    augmentedLed = BUILTIN_LED_OFF;
    connectivityLed = BUILTIN_LED_ON;
}

int main()
{
    pc.baud(115200);
    acc.set_defaults();

    // SimpleClient gives us an event queue which we can use, running on a separate thread (see https://docs.mbed.com/docs/mbed-os-handbook/en/5.1/concepts/events/)
    EventQueue* eventQueue = client.eventQueue();

    // Handle interrupts on the eventQueue - so we don't do things in interrupt context
    btn.fall(eventQueue->event(&fall));

    // Toggle the built-in LED every second (until we're connected, see `registered` function)
    connectivityTicker.attach(eventQueue->event(&toggleConnectivityLed), 1.0f);

    readSensorTicker.attach(eventQueue->event(&readSensor), 3.0f);

    // Functions can be executed through mbed Device Connector via POST calls (also defer to the event thread, so we never block)
    client.define_function("led/0/play", eventQueue->event(&play));
    
    // Connect to the internet (see mbed_app.json for the connectivity method)
    NetworkInterface* network = easy_connect(true);
    if (!network) {
        return 1;
    }

    // Connect to mbed Device Connector
    struct MbedClientOptions opts = client.get_default_options(); // opts contains information like the DeviceType
    bool setup = client.setup(opts, network);
    if (!setup) {
        pc.printf("Client setup failed\n");
        return 1;
    }
    client.on_registered(&registered);
}
