#include "mbed.h"
#include "security.h"
#include "easy-connect.h"
#include "simple-mbed-client.h"
#include "C12832.h"
#include "FXOS8700Q.h"

Serial pc(USBTX, USBRX);
C12832 lcd(D11, D13, D12, D7, D10);
I2C i2c(PTE25, PTE24);
FXOS8700QAccelerometer acc(i2c, FXOS8700CQ_SLAVE_ADDR1);
FXOS8700QMagnetometer mag(i2c, FXOS8700CQ_SLAVE_ADDR1);

SimpleMbedClient client;

// Declare peripherals
DigitalOut connectivityLed(LED2);                   // Blinks while connecting, turns solid when connected
DigitalOut augmentedLed(LED1, BUILTIN_LED_OFF);     // LED that can be controlled through Connector
InterruptIn btn(MBED_CONF_APP_BUTTON);              // Button that sends it's count to Connector

Ticker connectivityTicker;
Ticker readAccelTicker;

// Callback function when the pattern resource is updated
void patternUpdated(string v)
{
    pc.printf("New pattern: %s\n", v.c_str());
}

void lcdTextUpdated(string v)
{
    if (v.length() > 30) {
        v.erase(v.begin() + 30, v.end());
    }
    pc.printf("New text is %s\r\n", v.c_str());

    lcd.cls();
    lcd.locate(0, 3);
    lcd.printf(v.c_str());
}

// Define resources here. They act as normal variables, but are exposed to the internet...
SimpleResourceInt btn_count = client.define_resource("button/0/clicks", 0, M2MBase::GET_ALLOWED);
SimpleResourceString pattern = client.define_resource("led/0/pattern", "500:500:500:500:500:500:500", &patternUpdated);
SimpleResourceString lcd_text = client.define_resource("lcd/0/text",
                                "Hello from the cloud", M2MBase::GET_PUT_ALLOWED, true, &lcdTextUpdated);

SimpleResourceInt aX = client.define_resource("accel/0/x", 0, M2MBase::GET_ALLOWED);
SimpleResourceInt aY = client.define_resource("accel/0/y", 0, M2MBase::GET_ALLOWED);
SimpleResourceInt aZ = client.define_resource("accel/0/z", 0, M2MBase::GET_ALLOWED);

void readAccel()
{
    motion_data_counts_t acc_data;

    acc.getAxis(acc_data);
    aX = acc_data.x;
    aY = acc_data.y;
    aZ = acc_data.z;

    pc.printf("ACC: X=%d Y=%d Z=%d\r\n", acc_data.x, acc_data.y, acc_data.z);
}

void fall()
{
    btn_count = btn_count + 1;
    pc.printf("Button count is now %d\r\n", static_cast<int>(btn_count));
}

void toggleConnectivityLed()
{
    connectivityLed = !connectivityLed;
}

void registered()
{
    pc.printf("Registered\r\n");

    connectivityTicker.detach();
    connectivityLed = BUILTIN_LED_ON;
}

void play(void* args)
{
    connectivityLed = BUILTIN_LED_OFF;

    // Parse the pattern string, and toggle the LED in that pattern
    string s = static_cast<string>(pattern);
    size_t i = 0;
    size_t pos = s.find(':');
    while (pos != string::npos) {
        wait_ms(atoi(s.substr(i, pos - i).c_str()));
        augmentedLed = !augmentedLed;

        i = ++pos;
        pos = s.find(':', pos);

        if (pos == string::npos) {
            wait_ms(atoi(s.substr(i, s.length()).c_str()));
            augmentedLed = !augmentedLed;
        }
    }

    augmentedLed = BUILTIN_LED_OFF;
    connectivityLed = BUILTIN_LED_ON;
}

int main()
{
    pc.baud(115200);
    lcdTextUpdated(static_cast<string>(lcd_text).c_str());
    acc.enable();

    // SimpleClient gives us an event queue which we can use, running on a separate thread (see https://docs.mbed.com/docs/mbed-os-handbook/en/5.1/concepts/events/)
    EventQueue* eventQueue = client.eventQueue();

    // Handle interrupts on the eventQueue - so we don't do things in interrupt context
    btn.fall(eventQueue->event(&fall));

    // Toggle the built-in LED every second (until we're connected, see `registered` function)
    connectivityTicker.attach(eventQueue->event(&toggleConnectivityLed), 1.0f);

    readAccelTicker.attach(eventQueue->event(&readAccel), 1.0f);

    // Functions can be executed through mbed Device Connector via POST calls (also defer to the event thread, so we never block)
    client.define_function("led/0/play", eventQueue->event(&play));

    // Connect to the internet (see mbed_app.json for the connectivity method)
    NetworkInterface* network = easy_connect(true);
    if (!network) {
        return 1;
    }

    // Connect to mbed Device Connector
    struct MbedClientOptions opts = client.get_default_options(); // opts contains information like the DeviceType
    bool setup = client.setup(opts, network);
    if (!setup) {
        pc.printf("Client setup failed\n");
        return 1;
    }
    client.on_registered(&registered);
}
