// ----------------------------------------------------------------------------
// Copyright 2016-2018 ARM Ltd.
//
// SPDX-License-Identifier: Apache-2.0
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//     http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.
// ----------------------------------------------------------------------------

#include "mbed.h"
#include "simple-mbed-cloud-client.h"
#include "SDBlockDevice.h"
#include "FATFileSystem.h"
#include "ESP8266Interface.h"
#include "MMA7660.h"
#include "LM75B.h"

#define WIFI_SSID "SSID"
#define WIFI_PSWD "PASSWORD"

// An event queue is a very useful structure to debounce information between contexts (e.g. ISR and normal threads)
// This is great because things such as network operations are illegal in ISR, so updating a resource in a button's fall() function is not allowed
EventQueue eventQueue;
Thread thread1;

// Storage implementation definition, currently using SDBlockDevice (SPI flash, DataFlash, and internal flash are also available)
/* K64 & K66 */
InterruptIn sw2(SW2);
DigitalOut led2(LED2);

SDBlockDevice sd(PTE3, PTE1, PTE2, PTE4);
FATFileSystem fs("sd", &sd);
ESP8266Interface net(D1, D0);
LM75B lm75b(I2C_SDA, I2C_SCL);     // temperature
MMA7660 mma7660(I2C_SDA, I2C_SCL); // accel

const int NUM_AXIS = 3;

// Declaring pointers for access to Mbed Cloud Client resources outside of main()
MbedCloudClientResource *button_res;
MbedCloudClientResource *pattern_res;
MbedCloudClientResource *temp_res;
MbedCloudClientResource *temp_unit_res;    
MbedCloudClientResource *accel_res[NUM_AXIS];
MbedCloudClientResource *acc_unit_res;

static bool button_pressed = false;
static int button_count = 0;

void button_press()
{
    button_pressed = true;
    ++button_count;
    button_res->set_value(button_count);
}

/**
 * PUT handler
 * @param resource The resource that triggered the callback
 * @param newValue Updated value for the resource
 */
void pattern_updated(MbedCloudClientResource *resource, m2m::String newValue)
{
    printf("PUT received, new value: %s\n", newValue.c_str());
}

/**
 * POST handler
 * @param resource The resource that triggered the callback
 * @param buffer If a body was passed to the POST function, this contains the data.
 *               Note that the buffer is deallocated after leaving this function, so copy it if you need it longer.
 * @param size Size of the body
 */
void blink_callback(MbedCloudClientResource *resource, const uint8_t *buffer, uint16_t size)
{
    printf("POST received. Going to blink LED pattern: %s\n", pattern_res->get_value().c_str());

    static DigitalOut augmentedLed(LED1); // LED that is used for blinking the pattern

    // Parse the pattern string, and toggle the LED in that pattern
    string s = std::string(pattern_res->get_value().c_str());
    size_t i = 0;
    size_t pos = s.find(':');
    while (pos != string::npos) {
        wait_ms(atoi(s.substr(i, pos - i).c_str()));
        augmentedLed = !augmentedLed;

        i = ++pos;
        pos = s.find(':', pos);

        if (pos == string::npos) {
            wait_ms(atoi(s.substr(i, s.length()).c_str()));
            augmentedLed = !augmentedLed;
        }
    }
}

/**
 * Notification callback handler
 * @param resource The resource that triggered the callback
 * @param status The delivery status of the notification
 */
void button_callback(MbedCloudClientResource *resource, const NoticationDeliveryStatus status)
{
    printf("Button notification, status %s (%d)\n", MbedCloudClientResource::delivery_status_to_string(status), status);
}

void temp_callback(MbedCloudClientResource *resource, const NoticationDeliveryStatus status)
{
    printf("Temperature notification, status %s (%d)\n", MbedCloudClientResource::delivery_status_to_string(status), status);
}

void accel_callback(MbedCloudClientResource *resource, const NoticationDeliveryStatus status)
{
    printf("Accelerometer notification, status %s (%d)\n", MbedCloudClientResource::delivery_status_to_string(status), status);
}

void measure_sensors()
{
    float temperature, acc[3];
    const unsigned int buf_size = 20;
    char buf[buf_size];

    mma7660.readData(acc);
    for(int i=0; i < NUM_AXIS; i++) {
        snprintf(buf, buf_size, "%f", acc[i]);
        accel_res[i]->set_value(buf);
    }
    printf("acc: %f,%f,%f\n", acc[0], acc[1], acc[2]);

    temperature = lm75b.read();
    snprintf(buf, buf_size, "%f", temperature);
    temp_res->set_value(buf);
    printf("temp: %s\n", buf);
}

/**
 * Registration callback handler
 * @param endpoint Information about the registered endpoint such as the name (so you can find it back in portal)
 */
void registered(const ConnectorClientEndpointInfo *endpoint)
{
    printf("Connected to Mbed Cloud. Endpoint Name: %s\n", endpoint->internal_endpoint_name.c_str());
}

int main(void)
{
    printf("Starting Simple Mbed Cloud Client example\n");
    printf("Connecting to the network using Wi-Fi...\n");

    // Connect to the internet (DHCP is expected to be on)
    nsapi_error_t status = net.connect(WIFI_SSID, WIFI_PSWD, NSAPI_SECURITY_WPA_WPA2);

    if (status != 0) {
        printf("Connecting to the network failed %d!\n", status);
        return -1;
    }

    printf("Connected to the network successfully. IP address: %s\n", net.get_ip_address());

    // SimpleMbedCloudClient handles registering over LwM2M to Mbed Cloud
    SimpleMbedCloudClient client(&net, &sd, &fs);
    int client_status = client.init();
    if (client_status != 0) {
        printf("Initializing Mbed Cloud Client failed (%d)\n", client_status);
        return -1;
    }

    // Creating resources, which can be written or read from the cloud
    button_res = client.create_resource("3200/0/5501", "button_count");
    button_res->set_value(0);
    button_res->methods(M2MMethod::GET);
    button_res->observable(true);
    button_res->attach_notification_callback(button_callback);

    pattern_res = client.create_resource("3201/0/5853", "blink_pattern");
    pattern_res->set_value("500:500:500:500:500:500:500:500");
    pattern_res->methods(M2MMethod::GET | M2MMethod::PUT);
    pattern_res->attach_put_callback(pattern_updated);

    MbedCloudClientResource *blink_res = client.create_resource("3201/0/5850", "blink_action");
    blink_res->methods(M2MMethod::POST);
    blink_res->attach_post_callback(blink_callback);
    
    temp_res = client.create_resource("3303/0/5700", "temperature");
    temp_res->set_value("0");
    temp_res->methods(M2MMethod::GET);
    temp_res->attach_notification_callback(temp_callback);
    temp_res->observable(true);

    temp_unit_res = client.create_resource("3303/0/5701", "unit");
    temp_unit_res->set_value("Cel");
    
    accel_res[0] = client.create_resource("3313/0/5702", "accel_x");
    accel_res[1] = client.create_resource("3313/0/5703", "accel_y");
    accel_res[2] = client.create_resource("3313/0/5704", "accel_z");

    for (int i=0; i < NUM_AXIS; i++) {
        accel_res[i]->set_value(0);
        accel_res[i]->methods(M2MMethod::GET);
        accel_res[i]->attach_notification_callback(accel_callback);
        accel_res[i]->observable(true);
    }

    acc_unit_res = client.create_resource("3313/0/5701", "unit");
    acc_unit_res->set_value("G");
    
    printf("Initialized Mbed Cloud Client. Registering...\n");

    // Callback that fires when registering is complete
    client.on_registered(&registered);

    // Register with Mbed Cloud
    client.register_and_connect();

    // Setup the button
    sw2.mode(PullUp);

    // The button fall handler is placed in the event queue so it will run in
    // thread context instead of ISR context, which allows safely updating the cloud resource
    sw2.fall(eventQueue.event(&button_press));
    button_count = 0;

    // Placeholder for callback to update local resource when GET comes.
    // The timer fires on an interrupt context, but debounces it to the eventqueue, so it's safe to do network operations
    Ticker timer;
    timer.attach(eventQueue.event(&measure_sensors), 5.0);

    // Start the event queue in a separate thread so the main thread continues
    thread1.start(callback(&eventQueue, &EventQueue::dispatch_forever));

    while(1) {
        wait_ms(100);

        if (button_pressed) {
            button_pressed = false;
            printf("button clicked %d times\r\n", button_count);
        }

    }
}
