#include "mbed.h"
#include "WiflyInterface.h"
#include "C12832_lcd.h"
#include "nsdl_support.h"
#include "dbg.h"
// Include various resources
#include "temperature.h"
#include "light.h"
#include "gps.h"
#include "relay.h"

static C12832_LCD lcd;
Serial pc(USBTX, USBRX); // tx, rx

// ****************************************************************************
// Configuration section

// Ethernet configuration
/* Define this to enable DHCP, otherwise manual address configuration is used */
#define DHCP

/* Manual IP configurations, if DHCP not defined */
#define IP      "10.45.0.206"
#define MASK    "255.255.255.0"
#define GW      "10.45.0.1"

// NSP configuration
/* Change this IP address to that of your NanoService Platform installation */
static const char* NSP_ADDRESS = "217.140.101.20"; /* demo NSP, web interface at http://nanoservice-demo.mbed.org*/ 
static const int NSP_PORT = 5683;
char endpoint_name[16] = "mbed-";
uint8_t ep_type[] = {"mbed_device"};
uint8_t lifetime_ptr[] = {"1200"};

// ****************************************************************************
// WiFly interface initialization

WiflyInterface wifly(p9, p10, p30, p29, "ssid", "password", WPA);

static void wifly_init()
{
    char mbed_uid[33]; // for creating unique name for the board

    /* Initialize network */
    NSDL_DEBUG("DHCP in use\r\n");
    wifly.init();

    while(!wifly.connect())
        ;
        
    pc.printf("Connect OK\n\r");

    mbed_interface_uid(mbed_uid);
    mbed_uid[32] = '\0';
    strncat(endpoint_name, mbed_uid + 27, 15 - strlen(endpoint_name));

    lcd.locate(0,11);
    lcd.printf("IP:%s", wifly.getIPAddress());

    NSDL_DEBUG("IP Address:%s ", wifly.getIPAddress());
}

// ****************************************************************************
// NSP initialization

UDPSocket server;
Endpoint nsp;

static void nsp_init()
{
    server.init();
    server.bind(NSP_PORT);

    nsp.set_address(NSP_ADDRESS, NSP_PORT);
    
    NSDL_DEBUG("name: %s", endpoint_name);
    NSDL_DEBUG("NSP=%s - port %d\n", NSP_ADDRESS, NSP_PORT);

    lcd.locate(0,22);
    lcd.printf("EP name:%s\n", endpoint_name);
}

// ****************************************************************************
// Resource creation

static int create_resources()
{
    sn_nsdl_resource_info_s *resource_ptr = NULL;
    sn_nsdl_ep_parameters_s *endpoint_ptr = NULL;
    
    NSDL_DEBUG("Creating resources");

    /* Create resources */
    resource_ptr = (sn_nsdl_resource_info_s*)nsdl_alloc(sizeof(sn_nsdl_resource_info_s));
    if(!resource_ptr)
        return 0;
    memset(resource_ptr, 0, sizeof(sn_nsdl_resource_info_s));

    resource_ptr->resource_parameters_ptr = (sn_nsdl_resource_parameters_s*)nsdl_alloc(sizeof(sn_nsdl_resource_parameters_s));
    if(!resource_ptr->resource_parameters_ptr)
    {
        nsdl_free(resource_ptr);
        return 0;
    }
    memset(resource_ptr->resource_parameters_ptr, 0, sizeof(sn_nsdl_resource_parameters_s));

    // Static resources
    nsdl_create_static_resource(resource_ptr, sizeof("dev/mfg")-1, (uint8_t*) "dev/mfg", 0, 0,  (uint8_t*) "Sensinode", sizeof("Sensinode")-1);
    nsdl_create_static_resource(resource_ptr, sizeof("dev/mdl")-1, (uint8_t*) "dev/mdl", 0, 0,  (uint8_t*) "NSDL-C mbed device", sizeof("NSDL-C mbed device")-1);

    // Dynamic resources
    create_temperature_resource(resource_ptr);
    create_light_resource(resource_ptr);
    create_gps_resource(resource_ptr);
    create_relay_resource(resource_ptr);

        /* Register with NSP */
    endpoint_ptr = nsdl_init_register_endpoint(endpoint_ptr, (uint8_t*)endpoint_name, ep_type, lifetime_ptr);
    if(sn_nsdl_register_endpoint(endpoint_ptr) != 0)
        pc.printf("NSP registering failed\r\n");
    else
        pc.printf("NSP registering OK\r\n");
    nsdl_clean_register_endpoint(&endpoint_ptr);

    nsdl_free(resource_ptr->resource_parameters_ptr);
    nsdl_free(resource_ptr);
    return 1;
}

// ****************************************************************************
// Program entry point

int main()
{
    lcd.cls();
    lcd.locate(0,0);
    lcd.printf("mbed NanoService demo");
    NSDL_DEBUG("mbed NanoService Example App 0.1\n");
    
    // Initialize Wifly interface first
    wifly_init();
    
    // Initialize NSP node
    nsp_init();
    
    // Initialize NSDL stack
    nsdl_init();
    
    // Create NSDL resources
    create_resources();
    
    // Run the NSDL event loop (never returns)
    nsdl_event_loop();
}
